package partnerships

import (
	"fmt"
	"time"

	"context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/xray"
	moneypenny "code.justin.tv/revenue/moneypenny/client"
	"github.com/afex/hystrix-go/hystrix"
)

func init() {
	hystrix.ConfigureCommand("Partnerships.CheckPartneredByUserID", hystrix.CommandConfig{
		Timeout:                3000,
		SleepWindow:            5000,
		ErrorPercentThreshold:  50,
		RequestVolumeThreshold: 20,
		MaxConcurrentRequests:  100,
	})
	hystrix.ConfigureCommand("Partnerships.GetUserPayoutType", hystrix.CommandConfig{
		Timeout:                3000,
		SleepWindow:            5000,
		ErrorPercentThreshold:  50,
		RequestVolumeThreshold: 20,
		MaxConcurrentRequests:  100,
	})
}

type HystrixClient struct {
	client Client
}

func (c *HystrixClient) CheckPartneredByUserID(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (bool, error) {
	var ret bool
	var err error

	hystrixErr := hystrix.Do("Partnerships.CheckPartneredByUserID", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Partnerships.CheckPartneredByUserID circuit panic=%v", r))
			}
		}()

		ret, err = c.client.CheckPartneredByUserID(ctx, userID, reqOpts)
		return err
	}, nil)

	if hystrixErr != nil {
		return false, hystrixErr
	}

	if err != nil {
		// If the context contains xray information, keep it
		dCtx := xray.DetachContext(ctx)
		// Ensure the timeout can't run forever
		tCtx, cancel := context.WithTimeout(dCtx, 10*time.Second)
		defer cancel()
		// Attach relevant fields from logx to detatched Context
		tCtx = logx.CopyFields(ctx, tCtx)

		logx.Error(tCtx, err)
	}

	return ret, err
}

func (c *HystrixClient) GetUserPayoutType(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*moneypenny.GetUserPayoutTypeResponse, error) {
	var ret *moneypenny.GetUserPayoutTypeResponse
	var err error

	hystrixErr := hystrix.Do("Partnerships.GetUserPayoutType", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Partnerships.GetUserPayoutType circuit panic=%v", r))
			}
		}()

		ret, err = c.client.GetUserPayoutType(ctx, userID, reqOpts)
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	if err != nil {
		// If the context contains xray information, keep it
		dCtx := xray.DetachContext(ctx)
		// Ensure the timeout can't run forever
		tCtx, cancel := context.WithTimeout(dCtx, 10*time.Second)
		defer cancel()
		// Attach relevant fields from logx to detatched Context
		tCtx = logx.CopyFields(ctx, tCtx)

		logx.Error(tCtx, err)
	}

	return ret, err
}
