package pubsub

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	pubclient "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"PubSub.Publish": {
			Timeout:               500,
			MaxConcurrentRequests: 1000,
		},
	})
}

//go:generate mockery -name PubSub
type PubSub interface {
	Publish(ctx context.Context, topic []string, event models.PubSubEvent) error
}

type pubSubImpl struct {
	pubClient pubclient.PubClient
}

func NewPubSub(host string) (PubSub, error) {
	pubsub, err := pubclient.NewPubClient(twitchclient.ClientConf{
		Host: host,
	})
	return &pubSubImpl{pubsub}, err
}

func (p *pubSubImpl) Publish(ctx context.Context, topic []string, event models.PubSubEvent) error {
	marshalled, err := json.Marshal(event)
	if err != nil {
		return err
	}

	return hystrix.Do("PubSub.Publish", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Errorf("circuit panic %v", p))
			}
		}()
		pubErr := p.pubClient.Publish(ctx, topic, string(marshalled), nil)
		return pubErr
	}, nil)
}
