package rails

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"bytes"
	"errors"

	"code.justin.tv/foundation/twitchclient"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

const (
	timingXactName = "service.rails"
)

func newStatName(metric string) string {
	return timingXactName + "." + metric
}

var ErrUserNotFound = errors.New("User not found on rails.")

type emailBody struct {
	UserID       string      `json:"user_id,omitempty"`
	EmailName    string      `json:"email_name"`
	EmailArgs    interface{} `json:"email_args"`
	EmailAddress string      `json:"email_address,omitempty"`
}

type ErrorResponse struct {
	Status  int
	Message string
	Error   string
}

// Client exposes methods to interact with the cache
//go:generate mockery -name Client
type Client interface {
	DeleteCache(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error
	SendEmail(ctx context.Context, ID string, emailName string, emailAddress string, emailArgs interface{}) error
	SoftDeleteUser(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error
	HardDeleteUser(ctx context.Context, ID string, adminLogin string, reqOpts *twitchclient.ReqOpts) error
	UndeleteUser(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error
	VodSync(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error
}

// NewClient creates a Cache client
func NewClient(host string, stats statsd.Statter) (Client, error) {
	c, err := twitchclient.NewClient(twitchclient.ClientConf{
		Host:           host,
		Stats:          stats,
		TimingXactName: timingXactName,
	})
	if err != nil {
		return nil, err
	}

	return &railsClientImpl{
		Client: c,
	}, nil
}

type railsClientImpl struct {
	twitchclient.Client
}

// DeleteCache removes an ID from the rails cache
func (b *railsClientImpl) DeleteCache(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error {
	epFormat := "/api/internal/cache/%s"
	endpoint := fmt.Sprintf(epFormat, ID)

	req, err := b.NewRequest("DELETE", endpoint, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       newStatName("delete_cache"),
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}
	return nil
}

func (b *railsClientImpl) SendEmail(ctx context.Context, ID string, emailName string, emailAddress string, emailArgs interface{}) error {
	emailBody := emailBody{
		EmailName: emailName,
		EmailArgs: emailArgs,
	}

	if ID != "" {
		emailBody.UserID = ID
	} else {
		emailBody.EmailAddress = emailAddress
	}

	body, err := json.Marshal(emailBody)
	if err != nil {
		return err
	}

	req, err := b.NewRequest("POST", "/api/internal/emails", bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       newStatName("send_email"),
		StatSampleRate: 1.0,
	}

	resp, err := b.Do(ctx, req, reqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode == 404 {
		return ErrUserNotFound
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}
	return nil
}

// handleFailedRequest is a generic error parser / formatter for all cache endpoints
func (b *railsClientImpl) handleFailedRequest(resp *http.Response) error {
	var errResp ErrorResponse
	err := json.NewDecoder(resp.Body).Decode(&errResp)
	if err != nil {
		return fmt.Errorf("Failed rails request. StatusCode=%v Unable to read response body (%v)", resp.StatusCode, err)
	}

	return fmt.Errorf(`Failed rails request. StatusCode=%v Message=%v`, resp.StatusCode, errResp.Message)
}

func (b *railsClientImpl) SoftDeleteUser(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{}
	query.Add("id", ID)

	path := (&url.URL{
		Path:     "/api/internal/soft_delete_user",
		RawQuery: query.Encode(),
	}).String()

	req, err := b.NewRequest("DELETE", path, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       newStatName("soft_delete_user"),
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode == 404 {
		return ErrUserNotFound
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}

	return nil
}

func (b *railsClientImpl) HardDeleteUser(ctx context.Context, ID string, adminLogin string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{}
	query.Add("id", ID)
	query.Add("admin_id", adminLogin)

	path := (&url.URL{
		Path:     "/api/internal/hard_delete_user",
		RawQuery: query.Encode(),
	}).String()

	req, err := b.NewRequest("DELETE", path, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       newStatName("hard_delete_user"),
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode == 404 {
		return ErrUserNotFound
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}

	return nil
}

func (b *railsClientImpl) UndeleteUser(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{}
	query.Add("id", ID)

	path := (&url.URL{
		Path:     "/api/internal/undelete_user",
		RawQuery: query.Encode(),
	}).String()

	req, err := b.NewRequest("PUT", path, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       newStatName("undelete_user"),
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode == 404 {
		return ErrUserNotFound
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}

	return nil
}

func (b *railsClientImpl) VodSync(ctx context.Context, ID string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{}
	query.Add("id", ID)

	path := (&url.URL{
		Path:     "/api/internal/vod/sync",
		RawQuery: query.Encode(),
	}).String()

	req, err := b.NewRequest("POST", path, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       newStatName("vod_sync"),
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode == 404 {
		return ErrUserNotFound
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}

	return nil
}
