package s3

import (
	"fmt"

	"golang.org/x/net/context"

	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"golang.org/x/sync/errgroup"
)

const (
	defaultBucket = "ttv-user-pictures-prod"
	oldBucket     = "jtv_user_pictures"
	// We have no permission to touch this bucket anyway, but extra to make sure we don't intend to touch this
	defaultImageBucket   = "user-default-pictures"
	hystrixTimeout       = 15000
	hystrixMaxConcurrent = 10000
	objectNotFound       = "NoSuchKey"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"S3Get": {
			Timeout:               hystrixTimeout,
			MaxConcurrentRequests: hystrixMaxConcurrent,
		},
		"S3Delete": {
			Timeout:               hystrixTimeout,
			MaxConcurrentRequests: hystrixMaxConcurrent,
		},
		"S3Copy": {
			Timeout:               hystrixTimeout,
			MaxConcurrentRequests: hystrixMaxConcurrent,
		},
	})
}

//go:generate mockery -name S3Client
type S3Client interface {
	BatchCopy(paths map[string]string) error
	BatchDelete(paths []string) error
}

type S3ClientImpl struct {
	client       *s3.S3
	legacyClient *s3.S3
}

type OldCredentialProvider struct {
	Key    string
	Secret string
}

func (o OldCredentialProvider) IsExpired() bool {
	return false
}

func (o OldCredentialProvider) Retrieve() (credentials.Value, error) {
	return credentials.Value{
		AccessKeyID:     o.Key,
		SecretAccessKey: o.Secret,
	}, nil
}

func New(o OldCredentialProvider) (S3Client, error) {
	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}

	credentials := credentials.NewCredentials(o)
	return &S3ClientImpl{
		client:       s3.New(sess, aws.NewConfig().WithRegion("us-west-2")),
		legacyClient: s3.New(sess, aws.NewConfig().WithRegion("us-east-1").WithCredentials(credentials)),
	}, nil
}

func (s *S3ClientImpl) get(key, bucket string, client *s3.S3) error {
	return hystrixGet(
		&s3.GetObjectInput{
			Bucket: aws.String(bucket),
			Key:    aws.String(key),
		}, client)
}

func (s *S3ClientImpl) copy(original string, destination string) error {
	bucket := defaultBucket
	c := s.client
	err := s.get(original, oldBucket, s.legacyClient)
	if err != nil {
		if err, ok := err.(awserr.Error); !ok || err.Code() != objectNotFound {
			return err
		}
	} else {
		bucket = oldBucket
		c = s.legacyClient
	}

	input := &s3.CopyObjectInput{
		Bucket:     aws.String(bucket),
		CopySource: getKey(bucket, original),
		Key:        getKey("", destination),
		ACL:        aws.String("public-read"),
	}

	err = hystrixCopy(input, c)
	return err
}

func hystrixCopy(input *s3.CopyObjectInput, client *s3.S3) error {
	return hystrix.Do("S3Copy", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Errorf("%s circuit panic %v", *input.CopySource, p))

			}
		}()
		_, copyErr := client.CopyObject(input)
		if copyErr != nil {
			logx.Error(context.Background(), "failed to s3 copy", logx.Fields{
				"error":  fmt.Sprintf("%v", copyErr),
				"bucket": *input.Bucket,
				"source": *input.CopySource,
			})
		}
		return copyErr
	}, nil)
}

func hystrixGet(input *s3.GetObjectInput, client *s3.S3) error {
	err := hystrix.Do("S3Get", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Errorf("%s circuit panic %v", *input.Key, p))

			}
		}()
		get, getErr := client.GetObject(input)
		if getErr == nil {
			defer func() {
				if cerr := get.Body.Close(); cerr != nil {
					logx.Error(context.Background(), "failed to close s3 get body", logx.Fields{
						"error":  cerr,
						"bucket": *input.Bucket,
					})
				}
			}()
		}
		return getErr
	}, nil)
	return err
}

func (s *S3ClientImpl) BatchDelete(paths []string) error {
	g := &errgroup.Group{}
	g.Go(func() error {
		return hystrixDelete(paths, defaultBucket, s.client)
	})
	g.Go(func() error {
		return hystrixDelete(paths, oldBucket, s.legacyClient)
	})

	return g.Wait()
}

func hystrixDelete(paths []string, bucket string, client *s3.S3) error {
	identifiers := []*s3.ObjectIdentifier{}
	for i := range paths {
		if !strings.Contains(paths[i], defaultImageBucket) {
			identifiers = append(identifiers, &s3.ObjectIdentifier{
				Key: aws.String(paths[i]),
			})
		}
	}
	input := &s3.DeleteObjectsInput{
		Bucket: aws.String(bucket),
		Delete: &s3.Delete{
			Objects: identifiers,
		},
	}
	return hystrix.Do("S3Delete", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Errorf("%s circuit panic %v", *input.Bucket, p))

			}
		}()
		_, deleteErr := client.DeleteObjects(input)
		return deleteErr
	}, nil)
}

func (s *S3ClientImpl) BatchCopy(paths map[string]string) error {
	g := &errgroup.Group{}
	for from, to := range paths {
		from, to := from, to
		g.Go(func() error {
			return s.copy(from, to)
		})
	}

	return g.Wait()
}

func getKey(bucket, name string) *string {
	key := fmt.Sprintf("%s/%s", bucket, name)
	return aws.String(key)
}
