package sns

import (
	"encoding/json"
	"net/http"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/internal/compare"
	"code.justin.tv/web/users-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

const (
	origin   = "web/users-service"
	snsError = "sns_err"
)

// Publisher publishes an event to an SNS topic, logging any errors that may
// have occurred
//go:generate mockery -name Publisher
type Publisher interface {
	PublishRename(ctx context.Context, e models.SNSRenameEvent) error
	PublishUnban(ctx context.Context, e models.SNSUnbanEvent) error
	PublishCreation(ctx context.Context, e models.SNSCreationEvent) error
	PublishBan(ctx context.Context, e models.SNSBanEvent) error
	PublishUpdate(ctx context.Context, e models.SNSUpdateEvent) error
	PublishChannelUpdate(ctx context.Context, e models.SNSChannelUpdateEvent) error
	PublishImageUpdate(ctx context.Context, e models.SNSUpdateImageEvent) error
	PublishSoftDelete(ctx context.Context, e models.SNSSoftDeleteEvent) error
	PublishHardDelete(ctx context.Context, e models.SNSHardDeleteEvent) error
	PublishUndelete(ctx context.Context, e models.SNSUndeleteEvent) error
	PublishBanUser(ctx context.Context, e models.SNSBanUserEvent) error
	PublishExpireCache(ctx context.Context, e models.SNSExpireCacheEvent) error
}

// Config specifies the variable settings for an SNS publisher
type Config struct {
	Regions                 []string
	UserModerationEventsARN string
	UserRenameEventsARN     string
	UserCreationEventARN    string
	UserMutationEventARN    string
	ChannelMutationEventARN string
	PushyDispatchEventARN   string
	UserSoftDeleteEventsARN string
	UserHardDeleteEventsARN string
	UserUndeleteEventsARN   string
	ExpireCacheEventsARN    string
	Credentials             *credentials.Credentials
}

// NewPublisher instantiates a new Publisher
func NewPublisher(c Config, stats statsd.Statter) (Publisher, error) {
	if c.Regions == nil || len(c.Regions) == 0 {
		return nil, errx.New("no regions passed in")
	}

	snsByRegion := make(map[string]snsiface.SNSAPI)
	for _, region := range c.Regions {
		httpClient := &http.Client{
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 100,
			},
		}

		sess := session.New(&aws.Config{
			HTTPClient:  httpClient,
			Region:      aws.String(region),
			Credentials: c.Credentials,
		})
		snsByRegion[region] = sns.New(sess)
	}

	return &publisherImpl{
		sns:   snsByRegion,
		stats: stats,
		userModerationEventsARN: c.UserModerationEventsARN,
		userRenameEventsARN:     c.UserRenameEventsARN,
		userCreationEventARN:    c.UserCreationEventARN,
		userMutationEventARN:    c.UserMutationEventARN,
		channelMutationEventARN: c.ChannelMutationEventARN,
		pushyDispatchEventARN:   c.PushyDispatchEventARN,
		userSoftDeleteEventsARN: c.UserSoftDeleteEventsARN,
		userHardDeleteEventsARN: c.UserHardDeleteEventsARN,
		userUndeleteEventsARN:   c.UserUndeleteEventsARN,
		expireCacheEventsARN:    c.ExpireCacheEventsARN,
	}, nil
}

type publisherImpl struct {
	sns                     map[string]snsiface.SNSAPI
	stats                   statsd.Statter
	userModerationEventsARN string
	userRenameEventsARN     string
	userCreationEventARN    string
	userMutationEventARN    string
	channelMutationEventARN string
	pushyDispatchEventARN   string
	userSoftDeleteEventsARN string
	userHardDeleteEventsARN string
	userUndeleteEventsARN   string
	expireCacheEventsARN    string
}

func (p *publisherImpl) PublishRename(ctx context.Context, e models.SNSRenameEvent) error {
	if e.Data.Changes == nil {
		return nil
	}

	diff := &models.SNSRenameProperties{}

	hasChanged, err := compare.ByFieldName(diff, e.Data.Changes, e.Data.Original)
	if err != nil {
		return err
	}

	if hasChanged {
		e.Data.Changes = diff
		return p.publish(ctx, p.userRenameEventsARN, "rename_user", e)
	}

	return nil
}

func (p *publisherImpl) PublishUnban(ctx context.Context, e models.SNSUnbanEvent) error {
	return p.publish(ctx, p.userModerationEventsARN, "unban_user", e)
}

func (p *publisherImpl) PublishBan(ctx context.Context, e models.SNSBanEvent) error {
	return p.publish(ctx, p.userModerationEventsARN, "ban_user", e)
}

func (p *publisherImpl) PublishCreation(ctx context.Context, e models.SNSCreationEvent) error {
	return p.publish(ctx, p.userCreationEventARN, "create_user", e)
}

func (p *publisherImpl) PublishUpdate(ctx context.Context, e models.SNSUpdateEvent) error {
	if e.Changed == nil {
		return nil
	}

	diff := &models.UpdateableProperties{}

	hasChanged, err := compare.ByFieldName(diff, e.Changed, e.Original)
	if err != nil {
		return err
	}

	if hasChanged {
		e.Changed = diff
		return p.publish(ctx, p.userMutationEventARN, "update_user", e)
	}

	return nil
}

func (p *publisherImpl) PublishImageUpdate(ctx context.Context, e models.SNSUpdateImageEvent) error {
	if e.Changed == nil {
		return nil
	}

	diff := &models.ImageProperties{}

	hasChanged, err := compare.ByFieldName(diff, e.Changed, e.Original)
	if err != nil {
		return err
	}

	if hasChanged {
		e.Changed = diff
		return p.publish(ctx, p.userMutationEventARN, "update_user_images", e)
	}

	return nil
}

func (p *publisherImpl) PublishChannelUpdate(ctx context.Context, e models.SNSChannelUpdateEvent) error {
	if e.Changes == nil {
		return nil
	}

	diff := &models.UpdateChannelProperties{}

	hasChanged, err := compare.ByFieldName(diff, e.Changes, e.Original)
	if err != nil {
		return err
	}

	if hasChanged {
		e.Changes = diff
		return p.publish(ctx, p.channelMutationEventARN, "update_channel", e)
	}

	return nil
}

func (p *publisherImpl) PublishBanUser(ctx context.Context, e models.SNSBanUserEvent) error {
	return p.publish(ctx, p.pushyDispatchEventARN, "ban_user", e)
}

func (p *publisherImpl) PublishSoftDelete(ctx context.Context, e models.SNSSoftDeleteEvent) error {
	return p.publish(ctx, p.userSoftDeleteEventsARN, "soft_delete_user", e)
}

func (p *publisherImpl) PublishHardDelete(ctx context.Context, e models.SNSHardDeleteEvent) error {
	return p.publish(ctx, p.userHardDeleteEventsARN, "hard_delete_user", e)
}

func (p *publisherImpl) PublishUndelete(ctx context.Context, e models.SNSUndeleteEvent) error {
	return p.publish(ctx, p.userUndeleteEventsARN, "undelete_user", e)
}

func (p *publisherImpl) PublishExpireCache(ctx context.Context, e models.SNSExpireCacheEvent) error {
	return p.publish(ctx, p.expireCacheEventsARN, "expire_cache", e)
}

// publish attempts to publish the given event to the SNS topic.
// Errors during publish will be reported to Rollbar.
func (p *publisherImpl) publish(ctx context.Context, snsARN, eventName string, data interface{}) error {
	bytes, err := json.Marshal(data)
	if err != nil {
		return errx.New(err, errx.Fields{snsError: "marshalling SNS payload"})
	}

	in := &sns.PublishInput{
		Message: aws.String(string(bytes)),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": &sns.MessageAttributeValue{
				StringValue: aws.String(eventName),
				DataType:    aws.String("String"),
			},
			"origin": &sns.MessageAttributeValue{
				StringValue: aws.String(origin),
				DataType:    aws.String("String"),
			},
		},
		TopicArn: aws.String(snsARN),
	}
	region, err := getRegionFromARN(snsARN)
	if err != nil {
		return errx.New(err, errx.Fields{snsError: "publishing to SNS"})
	}

	snsClient, found := p.sns[region]
	if !found {
		return errx.New(err, errx.Fields{snsError: "No SNS client found for region: " + region})
	}
	_, err = snsClient.Publish(in)
	if err != nil {
		return errx.New(err, errx.Fields{snsError: "publishing to SNS"})
	}
	return nil
}

func getRegionFromARN(snsARN string) (string, error) {
	s := strings.Split(snsARN, ":")
	if len(s) < 4 {
		return "", errx.New("Malformed TopicARN")
	}
	return s[3], nil
}
