package twilio

import (
	"context"
	"errors"
	"hash/fnv"
	"strings"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/models"

	"github.com/afex/hystrix-go/hystrix"

	"bitbucket.org/ckvist/twilio/twirest"
	validator "gopkg.in/validator.v2"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"send_sms": {
			Timeout:               5000,
			MaxConcurrentRequests: 10000,
		},
	})
}

var ErrInvalidPhoneNumber = errors.New("invalid phone number")

type Config struct {
	AccountSSID string   `validate:"nonzero"`
	AuthToken   string   `validate:"nonzero"`
	FromChoices []string `validate:"min=1,nonzero"`
}

func (c Config) Validate() error {
	return validator.Validate(c)
}

//go:generate mockery -name Client
type Client interface {
	SendSMS(ctx context.Context, number, body string) error
}

func NewClient(config Config) (*client, error) {
	if err := config.Validate(); err != nil {
		return nil, err
	}

	twi := twirest.NewClient(config.AccountSSID, config.AuthToken)

	return &client{
		twi:         twi,
		fromChoices: config.FromChoices,
	}, nil
}

type client struct {
	twi         *twirest.TwilioClient
	fromChoices []string
}

func (c *client) number(to string) string {
	h := fnv.New32a()
	var sum int
	if _, err := h.Write([]byte(to)); err != nil {
		sum = 0
	} else {
		sum = int(h.Sum32())
	}

	return c.fromChoices[sum%len(c.fromChoices)]
}

func (c *client) SendSMS(ctx context.Context, number, body string) error {
	if number == "" {
		return ErrInvalidPhoneNumber
	}

	from := c.number(number)

	msg := twirest.SendMessage{
		Text: body,
		From: from,
		To:   number,
	}

	start := time.Now()
	err := hystrix.Do("send_sms", func() error {
		_, err := c.twi.Request(msg)
		return err
	}, nil)

	if isInvalidPhoneNumber(err) {
		err = models.ErrInvalidPhoneNumber
	}

	duration := time.Since(start)

	status := "success"
	if err != nil {
		status = "failure"
	}

	if metricErr := config.Statsd().TimingDuration("twilio.send_sms."+status, duration, 0.1); metricErr != nil {
		logx.Warn(ctx, "failed to send twilio.send_sms metric", logx.Fields{
			"err": err,
		})
	}

	return err
}

func isInvalidPhoneNumber(err error) bool {
	if err != nil && strings.Contains(err.Error(), "is not a valid phone number") {
		return true
	}

	if err != nil && strings.Contains(err.Error(), "is not a mobile number") {
		return true
	}

	return false
}
