package uploader

import (
	"fmt"
	"strconv"
	"strings"

	"encoding/json"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/common/yimg"
	"code.justin.tv/foundation/twitchclient"
	rpcuploader "code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"
)

const (
	defaultS3Path        = "s3://ttv-user-pictures-prod/"
	canonicalUserID      = "id=baaa022bae62837aa6780310f7096b0d140dc1b4ca5cdb6543af02bf7b4e96ec"
	globalReadAccessURI  = "uri=http://acs.amazonaws.com/groups/global/AllUsers"
	defaultImageType     = "png"
	compressionImageType = "jpg"
	gifImageType         = "gif"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"Uploader.UploadImages": {
			Timeout:               500,
			MaxConcurrentRequests: 1000,
		},
		"Uploader.UpdateUploadStatus": {
			Timeout:               500,
			MaxConcurrentRequests: 1000,
		},
	})
}

type Uploader interface {
	UpdateStatus(ctx context.Context, uploadID string, status rpcuploader.Status) error
	UploadImages(ctx context.Context, uploadableImage models.UploadableImage) (models.UploadInfo, error)
}

type uploaderImpl struct {
	uploaderClient  rpcuploader.Uploader
	imageSnsArn     string
	transformations map[string][]*rpcuploader.Output
}

func NewUploader(addr, sns string) (Uploader, error) {
	transformations := map[string][]*rpcuploader.Output{}

	httpCli := twitchclient.NewHTTPClient(twitchclient.ClientConf{})
	twirpCli := rpcuploader.NewUploaderProtobufClient(addr, httpCli)

	u := &uploaderImpl{twirpCli, sns, transformations}
	var err error
	err = u.readImages(models.ProfileBannerImageType)
	if err != nil {
		return nil, err
	}

	err = u.readImages(models.ChannelOfflineImageType)
	if err != nil {
		return nil, err
	}

	err = u.readImages(models.ProfilePictureImageType)
	if err != nil {
		return nil, err
	}

	return u, nil
}

func (u *uploaderImpl) UpdateStatus(ctx context.Context, uploadID string, status rpcuploader.Status) error {
	request := &rpcuploader.SetStatusRequest{
		UploadId: uploadID,
		Status:   status,
	}

	return hystrix.Do("Uploader.UpdateUploadStatus", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Sprintf("circuit panic %v", p))
			}
		}()
		_, setStatusErr := u.uploaderClient.SetStatus(ctx, request)
		return setStatusErr
	}, nil)
}

func (u *uploaderImpl) UploadImages(ctx context.Context, uploadableImage models.UploadableImage) (models.UploadInfo, error) {
	data := UserImageMetadata{
		ImageType: uploadableImage.Type,
		UserID:    uploadableImage.ID,
	}

	outputs := []*rpcuploader.Output{}
	format := defaultImageType
	if uploadableImage.Format != "" && uploadableImage.Format != defaultImageType && uploadableImage.Format != gifImageType {
		format = compressionImageType
	}

	data.Format = format

	uploadInfo := models.UploadInfo{}
	bytes, err := json.Marshal(data)
	if err != nil {
		return uploadInfo, err
	}

	callback := &rpcuploader.Callback{
		SnsTopicArn: u.imageSnsArn,
		Data:        bytes,
	}

	reformat := &transformations.Transcode{Format: format}
	trans := u.transformations[uploadableImage.Type]
	for _, output := range trans {
		newOutput := *output
		newOutput.Transformations = append([]*rpcuploader.Transformation{reformat.AsProto()}, newOutput.Transformations...)
		outputs = append(outputs, &newOutput)
	}
	request := &rpcuploader.UploadRequest{
		Outputs:       outputs,
		PreValidation: &rpcuploader.Validation{FileSizeLessThan: "10MB", Format: "image"},
		OutputPrefix:  defaultS3Path,
		Callback:      callback,
	}

	var url string
	var id string
	err = hystrix.Do("Uploader.UploadImages", func() (e error) {
		defer func() {
			if p := recover(); p != nil {
				e = errx.New(fmt.Sprintf("circuit panic %v", p))
			}
		}()
		resp, updateErr := u.uploaderClient.Create(ctx, request)
		if updateErr == nil {
			url = resp.GetUrl()
			id = resp.GetUploadId()
		}
		return updateErr
	}, nil)
	uploadInfo.ID = id
	uploadInfo.URL = url
	return uploadInfo, err
}

func (u *uploaderImpl) readImages(imageType string) error {
	var sizes []string
	var ratio float64
	switch imageType {
	case models.ProfilePictureImageType:
		sizes = yimg.ProfileImageSizes
		ratio = yimg.ProfileImageRatio
	case models.ProfileBannerImageType:
		sizes = yimg.ProfileBannerSizes
		ratio = 0
	case models.ChannelOfflineImageType:
		sizes = yimg.ChannelOfflineImageSizes
		ratio = yimg.ChannelOfflineImageRatio
	}

	for _, size := range sizes {
		var height, width int
		var err error
		if !strings.Contains(size, "x") {
			height, err = strconv.Atoi(size)
			if err != nil {
				return err
			}
		} else {
			dims := strings.Split(size, "x")

			width, err = strconv.Atoi(dims[0])
			if err != nil {
				return err
			}

			height, err = strconv.Atoi(dims[1])
			if err != nil {
				return err
			}
		}

		u.transformations[imageType] = append(u.transformations[imageType], newTransformation(ratio, width, height, imageType))
	}

	return nil
}

func newTransformation(ratio float64, width, height int, t string) *rpcuploader.Output {
	permissions := &rpcuploader.Permissions{
		GrantFullControl: canonicalUserID,
		GrantRead:        globalReadAccessURI,
	}

	var resize *transformations.ResizeDimensions

	if width != 0 {
		resize = &transformations.ResizeDimensions{Width: uint(width), Height: uint(height)}
	} else {
		resize = &transformations.ResizeDimensions{Height: uint(height)}
	}

	output := &rpcuploader.Output{
		Transformations: []*rpcuploader.Transformation{},
		Permissions:     permissions,
	}

	if t == models.ProfilePictureImageType {
		output.Name = fmt.Sprintf("{{UploadID}}-%s-{{Dimensions}}{{Extension}}", t)
	} else if t == models.ProfileBannerImageType {
		output.Name = fmt.Sprintf("{{UploadID}}-%s-{{Height}}{{Extension}}", t)
	} else if t == models.ChannelOfflineImageType {
		output.Name = fmt.Sprintf("{{UploadID}}-%s{{Extension}}", t)
	}

	if ratio != 0 {
		ratioTrans := &transformations.AspectRatio{ratio}
		output.Transformations = append(output.Transformations, ratioTrans.AsProto())
	}

	output.Transformations = append(output.Transformations, resize.AsProto())

	return output
}
