package compare

import (
	"errors"
	"fmt"
	"reflect"

	"code.justin.tv/chat/golibs/errx"
)

func reflectStruct(e interface{}) (reflect.Value, error) {
	value := reflect.ValueOf(e)

	// Follow pointer if necessary
	if value.Kind() == reflect.Ptr {
		value = value.Elem()
	}

	// Ensure target is a struct
	if value.Kind() != reflect.Struct {
		return reflect.Value{}, errors.New("expected target to be struct")
	}

	return value, nil
}

// ByFieldName compares changes to original looking for updates in changes. If differences
// are present, hasChanged is true otherwise false. Any updates in changes will be present in target.
// target and changes must be the same type. Comparison is done by field names.
func ByFieldName(target, changes, original interface{}) (hasChanged bool, err error) {
	defer func() {
		if e := recover(); e != nil {
			err = errx.New(fmt.Errorf("error in models.CompareByFieldName: %v", e))
		}
	}()

	// Get values of target and original for comparison.
	targetValue, err := reflectStruct(target)
	if err != nil {
		return false, err
	}
	changesValue, err := reflectStruct(changes)
	if err != nil {
		return false, err
	}
	originalValue, err := reflectStruct(original)
	if err != nil {
		return false, err
	}

	// Get target type so we can access field names
	targetType := targetValue.Type()
	changesType := changesValue.Type()

	if targetType != changesType {
		return false, errors.New("diff and changes must have same type")
	}

	for i := 0; i < changesType.NumField(); i++ {
		// Get field type so we can access it's name
		fieldType := changesType.Field(i)

		// Skip comparing ID
		if fieldType.Name == "ID" {
			continue
		}

		fieldName := fieldType.Name
		if name, ok := fieldType.Tag.Lookup("compareTo"); ok {
			fieldName = name
		}

		changesInterface := changesValue.Field(i)
		targetInterface := targetValue.FieldByName(fieldName)
		originalInterface := originalValue.FieldByName(fieldName)

		if !originalInterface.IsValid() {
			// original interface doesn't have this field
			continue
		}

		// If no change for this field, skip
		if changesInterface.IsNil() {
			continue
		}

		// Defaults to original value. May change
		currentValue := originalInterface

		// Handle case where changes is a pointer and original is not
		if changesInterface.Kind() == reflect.Ptr && originalInterface.Kind() != reflect.Ptr {
			// Get pointer to original type
			currentValue = reflect.New(originalInterface.Type())

			if changesInterface.Kind() != currentValue.Kind() {
				return false, errors.New("different types")
			}

			// Set value of pointer
			currentValue.Elem().Set(originalInterface)
		}

		// Determine if target and current value have changed
		changed := !reflect.DeepEqual(changesInterface.Interface(), currentValue.Interface())
		hasChanged = hasChanged || changed

		// Set or zero the value
		if changed {
			targetInterface.Set(changesInterface)
		} else {
			targetInterface.Set(reflect.Zero(fieldType.Type))
		}
	}

	// Ensure that ID is set on target with original value
	targetIDField := targetValue.FieldByName("ID")
	originalIDField := originalValue.FieldByName("ID")
	if targetIDField.CanSet() && originalIDField.IsValid() {
		targetIDField.Set(originalIDField)
	}

	return
}
