package compare

import (
	"testing"

	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
)

func strPtr(s string) *string {
	return &s
}

func TestByFieldName_Changed(t *testing.T) {
	original := &models.ChannelProperties{
		ID:   1,
		Game: strPtr("game1"),
	}
	update := &models.UpdateChannelProperties{
		ID:   1,
		Game: strPtr("game2"),
	}

	diff := &models.UpdateChannelProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.True(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.Game)
	assert.NotNil(t, update.Game)
	assert.NotNil(t, diff.Game)
	assert.Equal(t, *update.Game, *diff.Game)
}

func TestByFieldName_NotChanged(t *testing.T) {
	original := &models.ChannelProperties{
		ID:   1,
		Game: strPtr("game1"),
	}
	update := &models.UpdateChannelProperties{
		ID:   1,
		Game: strPtr("game1"),
	}

	diff := &models.UpdateChannelProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.False(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.Game)
	assert.NotNil(t, update.Game)
	assert.Nil(t, diff.Game)
}

func TestByFieldName_NotChangedZero(t *testing.T) {
	original := &models.ChannelProperties{
		ID:   1,
		Game: strPtr("game1"),
	}
	update := &models.UpdateChannelProperties{
		ID: 1,
	}

	diff := &models.UpdateChannelProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.False(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.Game)
	assert.Nil(t, update.Game)
	assert.Nil(t, diff.Game)
}

func TestByFieldName_NonStruct(t *testing.T) {
	original := strPtr("game1")
	update := strPtr("game2")

	hasChanged, err := ByFieldName(update, update, original)
	assert.NotNil(t, err)
	assert.False(t, hasChanged)
}

func TestByFieldName_StructInStructChanged(t *testing.T) {
	original := &models.Properties{
		ID: "1",
		ProfileImage: &yimg.Images{
			"1x1": {
				Width:  1,
				Height: 1,
				URL:    "http://here",
			},
		},
	}
	update := &models.ImageProperties{
		ID: "1",
		ProfileImage: &yimg.Images{
			"1x1": {
				Width:  1,
				Height: 1,
				URL:    "http://there",
			},
		},
	}

	diff := &models.ImageProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.True(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.ProfileImage)
	assert.NotNil(t, update.ProfileImage)
	assert.NotNil(t, diff.ProfileImage)
}

func TestByFieldName_StructInStructNotChanged(t *testing.T) {
	original := &models.Properties{
		ID: "1",
		ProfileImage: &yimg.Images{
			"1x1": {
				Width:  1,
				Height: 1,
				URL:    "http://here",
			},
		},
	}
	update := &models.ImageProperties{
		ID: "1",
		ProfileImage: &yimg.Images{
			"1x1": {
				Width:  1,
				Height: 1,
				URL:    "http://here",
			},
		},
	}

	diff := &models.ImageProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.False(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.ProfileImage)
	assert.NotNil(t, update.ProfileImage)
	assert.Nil(t, diff.ProfileImage)
}

func TestByFieldName_Nested(t *testing.T) {
	event := &models.SNSRenameEvent{
		UserID: "1",
		Data: &models.SNSRenameData{
			Original: &models.SNSRenameProperties{
				Login:       strPtr("asdf"),
				Displayname: strPtr("fdsa"),
			},
			Changes: &models.SNSRenameProperties{
				Login:       strPtr("fdsa"),
				Displayname: strPtr("fdsa"),
			},
		},
	}

	hasChanged, err := ByFieldName(event.Data.Changes, event.Data.Changes, event.Data.Original)
	assert.Nil(t, err)
	assert.True(t, hasChanged)
	assert.NotNil(t, event.Data.Original.Login)
	assert.NotNil(t, event.Data.Changes.Login)
	assert.NotNil(t, event.Data.Original.Displayname)
	assert.Nil(t, event.Data.Changes.Displayname)
}

func TestByFieldName_Tag(t *testing.T) {
	original := &models.Properties{
		ID:    "1",
		Login: strPtr("asdf"),
	}
	update := &models.UpdateableProperties{
		ID:       "1",
		Login:    strPtr("asdf"),
		NewLogin: strPtr("fdsa"),
	}

	diff := &models.UpdateableProperties{}

	hasChanged, err := ByFieldName(diff, update, original)
	assert.Nil(t, err)
	assert.True(t, hasChanged)
	assert.Equal(t, original.ID, update.ID)
	assert.Equal(t, original.ID, diff.ID)
	assert.NotNil(t, original.Login)
	assert.NotNil(t, update.NewLogin)
	assert.NotNil(t, diff.Login)
	assert.Equal(t, *update.NewLogin, *diff.Login)
}
