package testutils

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
)

var (
	errHasDifference = errors.New("contents are different")
)

// DiffFiles diffs files using git
func DiffFiles(a, b string) ([]byte, error) {
	cmd := exec.Command("git", "diff", "--no-index", "--no-renames", "--", a, b)
	cmdBytes, err := cmd.Output()
	if err != nil {
		isExitCodeOne := strings.Contains(err.Error(), "exit status 1")
		if !isExitCodeOne {
			return cmdBytes, err
		}

		err = errHasDifference
	}

	var newLineBuf bytes.Buffer
	fmt.Fprintln(&newLineBuf)
	cutLines := 5
	lines := strings.SplitAfterN(string(cmdBytes), newLineBuf.String(), cutLines)
	if len(lines) < cutLines {
		return cmdBytes, err
	}
	return []byte(lines[cutLines-1]), err
}

func DiffInterfaces(a, b interface{}) ([]byte, error) {
	cleanupFile := func(f *os.File) {
		err := f.Close()
		if err != nil {
			log.Println("failed to close", f.Name(), err)
		}
		err = os.Remove(f.Name())
		if err != nil {
			log.Println("failed to remove", f.Name(), err)
		}
	}

	tempFile := func(a interface{}) (*os.File, error) {
		b, err := json.MarshalIndent(a, "", "  ")
		if err != nil {
			return nil, err
		}

		f, err := ioutil.TempFile("", "diff-interfaces")
		if err != nil {
			return nil, err
		}

		_, err = f.Write(b)
		if err != nil {
			cleanupFile(f)
			return nil, err
		}

		return f, nil
	}

	aF, err := tempFile(a)
	if err != nil {
		return nil, err
	}

	defer cleanupFile(aF)

	bF, err := tempFile(b)
	if err != nil {
		return nil, err
	}

	defer cleanupFile(bF)
	return DiffFiles(aF.Name(), bF.Name())
}
