package testutils

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"testing"

	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

const directory = "code.justin.tv/web/users-service/internal/testutils/data/"

var TestDataPath = os.Getenv("GOPATH") + "/src/" + directory

func AssertEqual(t *testing.T, expected, actual interface{}) {
	b, err := equality(expected, actual)
	if err != nil {
		assert.FailNow(t, err.Error())
	}

	if b != nil {
		assert.FailNow(t, string(b))
	}
}

func Equal(expected, actual interface{}) bool {
	b, err := equality(expected, actual)
	if err != nil {
		fmt.Println(err)
		return false
	}

	if b != nil {
		fmt.Println(string(b))
		return false
	}

	return true
}

func equality(expected, actual interface{}) ([]byte, error) {
	if expected == nil || actual == nil {
		return nil, errors.New("nil types are different")
	}

	e, err := json.Marshal(expected)
	if err != nil {
		return nil, err
	}

	a, err := json.Marshal(actual)
	if err != nil {
		return nil, err
	}

	equal := string(e) == string(a)
	if !equal {
		result, err := DiffInterfaces(expected, actual)
		if err != nil && err == errHasDifference { // it's a difference
			return result, nil
		} else if err != nil {
			return nil, errors.Wrap(err, "unable to print interface diff:")
		}
		return result, err
	}
	return nil, nil
}

func ChannelEqual(expected models.ChannelProperties, actual channels.Channel) bool {
	c, err := json.Marshal(actual)
	if err != nil {
		return false
	}

	cp := models.ChannelProperties{}
	err = json.Unmarshal(c, &cp)
	if err != nil {
		return false
	}

	return Equal(expected, cp)
}

func ReadToStruct(path, name string, o interface{}) []byte {
	data, err := ioutil.ReadFile(path + name)
	if err != nil {
		log.Fatalf("Error reading file %s: %v", name, err)
	}
	err = json.Unmarshal(data, o)
	if err != nil {
		log.Fatalf("Error un-marshalling: %v", err)
	}
	return data
}

func CopyUser(prop *models.Properties) *models.Properties {
	return &models.Properties{
		ID:                      prop.ID,
		Login:                   prop.Login,
		Birthday:                prop.Birthday,
		DmcaViolation:           prop.DmcaViolation,
		TermsOfServiceViolation: prop.TermsOfServiceViolation,
		DeletedOn:               prop.DeletedOn,
		Language:                prop.Language,
		Category:                prop.Category,
		RemoteIP:                prop.RemoteIP,
		Email:                   prop.Email,
		LastLogin:               prop.LastLogin,
		BannedUntil:             prop.BannedUntil,
		DmcaViolationCount:      prop.DmcaViolationCount,
		TosViolationCount:       prop.TosViolationCount,
		Admin:                   prop.Admin,
		Subadmin:                prop.Subadmin,
		GlobalMod:               prop.GlobalMod,
		Displayname:             prop.Displayname,
		Description:             prop.Description,
		ProfileImage:            prop.ProfileImage,
		UpdatedOn:               prop.UpdatedOn,
		CreatedOn:               prop.CreatedOn,
		EmailVerified:           prop.EmailVerified,
		PhoneNumber:             prop.PhoneNumber,
		LastLoginChangeDate:     prop.LastLoginChangeDate,
	}
}

func CopyChannel(prop models.ChannelProperties) models.ChannelProperties {
	return models.ChannelProperties{
		ID:                       prop.ID,
		Name:                     prop.Name,
		DirectoryHidden:          prop.DirectoryHidden,
		Broadcaster:              prop.Broadcaster,
		BroadcasterLanguage:      prop.BroadcasterLanguage,
		BroadcasterSoftware:      prop.BroadcasterSoftware,
		CreatedOn:                prop.CreatedOn,
		Game:                     prop.Game,
		GameID:                   prop.GameID,
		Mature:                   prop.Mature,
		Status:                   prop.Status,
		Title:                    prop.Title,
		UpdatedOn:                prop.UpdatedOn,
		ViewsCount:               prop.ViewsCount,
		BlockNonPublicAds:        prop.BlockNonPublicAds,
		PrerollsDisabled:         prop.PrerollsDisabled,
		PostrollsDisabled:        prop.PostrollsDisabled,
		FightAdBlock:             prop.FightAdBlock,
		LastBroadcastTime:        prop.LastBroadcastTime,
		LastBroadcastID:          prop.LastBroadcastID,
		LastLiveNotificationSent: prop.LastLiveNotificationSent,
		About:                        prop.About,
		RedirectChannel:              prop.RedirectChannel,
		PrimaryTeamID:                prop.PrimaryTeamID,
		DisableChat:                  prop.DisableChat,
		ChannelOfflineImage:          prop.ChannelOfflineImage,
		ProfileBanner:                prop.ProfileBanner,
		ProfileBannerBackgroundColor: prop.ProfileBannerBackgroundColor,
	}
}

func GetBirthday(id string) models.Birthday {
	return models.Birthday{
		Day:   (*Users[id].Birthday).Day(),
		Month: (*Users[id].Birthday).Month(),
		Year:  (*Users[id].Birthday).Year(),
	}
}

func GetStringPointer(s string) *string {
	if s == "" {
		return nil
	} else {
		return &s
	}
}

func GetBoolPointer(b bool) *bool {
	return &b
}

func GetUint64Pointer(b uint64) *uint64 {
	return &b
}
