package deadletterqueue

import (
	"fmt"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	rpcuploader "code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/imageuploadqueue"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
)

func Run(ctx context.Context, c worker.Clients, msg snsmodels.Message) (err error) {
	defer func() {
		if r := recover(); r != nil {
			err = errx.New(fmt.Sprintf("deadletterqueue.Run circuit panic=%v", r))
		}
	}()

	switch msg.TopicARN {
	case c.Topics[worker.ImageUploadQueue]:
		err := handleFailedImageUploadMessage(ctx, c, msg)
		logx.Error(ctx, "bad image upload sqs message", logx.Fields{
			"unexpected error": err,
			"message_body":     msg,
		})

	case c.Topics[worker.EmailValidationQueue]:
		logx.Error(ctx, "bad email verification sqs message", logx.Fields{
			"message_body": msg,
		})
	default:
		logx.Error(ctx, "unkonwn sns message", logx.Fields{
			"message_body": msg,
		})
	}

	return nil
}

func handleFailedImageUploadMessage(ctx context.Context, c worker.Clients, msg snsmodels.Message) error {
	if metricErr := config.ClusterStatsd().Inc("user_image_upload.failure", 1, 0.1); metricErr != nil {
		logx.Warn(ctx, fmt.Sprintf("error incrementing user_image_upload.failure: %s", metricErr))
	}

	param, err := imageuploadqueue.FetchParam(ctx, msg)
	if err != nil {
		return err
	}

	if err := c.Uploader.UpdateStatus(ctx, param.UploadID, rpcuploader.Status_FEATURE_SERVICE_FAILED); err != nil {
		return err
	}

	return nil
}
