package emailvalidationsuccess

import (
	"encoding/json"
	"fmt"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/growth/emailvalidator/evs/documents"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"code.justin.tv/web/users-service/logic"
	"code.justin.tv/web/users-service/models"
)

const (
	// EventName is the name of the event which this handler handles.
	EventName = "emailvalidationsuccess"

	// throttleTTL defines how long before we can send another notification to an
	// address for an identical verification request (prevent spamming a user by clicking a
	// "resend" button constantly)
	throttleTTL = 10 * time.Minute
)

// Run is the function which should be called to handle the email status updates requests
func Run(ctx context.Context, c worker.Clients, msg snsmodels.Message) (err error) {
	defer func() {
		if r := recover(); r != nil {
			err = errx.New(fmt.Sprintf("emailvalidationsuccess.Run circuit panic=%v", r))
		}
	}()

	eventAttr, ok := msg.MessageAttributes["event"]
	if !ok || eventAttr.Type != "String" {
		return errx.New("missing or malformed event", errx.Fields{"event_attr": eventAttr})
	}
	event := eventAttr.Value

	ctx = logx.WithFields(ctx, logx.Fields{"users_service_event": event})

	eventParams := worker.EventParams{
		Data:      []byte(msg.Message),
		Timestamp: msg.Timestamp,
		EventName: event,
	}

	params, err := fetchParams(ctx, eventParams.Data)
	if err != nil {
		return err
	}

	namespace := params.Namespace
	if namespace == logic.EmailValidationNamespace {
		userID := params.Key
		currUser, err := c.Users.GetUserPropertiesByID(ctx, userID, nil)
		if err != nil {
			return errx.New("Could not fetch current Email")

		}
		currEmail := ""
		if currUser.Email != nil {
			currEmail = *currUser.Email
		}
		newEmail := params.Email
		verified := newEmail == currEmail
		upToUpdate := &models.UpdateableProperties{
			EmailVerified: &verified,
		}
		return c.Users.SetUserProperties(ctx, userID, upToUpdate)
	}

	return nil
}

func fetchParams(ctx context.Context, eventData []byte) (*documents.EmailValidationSuccessEventParams, error) {
	var params documents.EmailValidationSuccessEventParams
	if err := json.Unmarshal(eventData, &params); err != nil {
		return nil, errx.New(fmt.Sprintf("Unable to convert json to object for event body: %v", err), errx.Fields{
			"event_body:": string(eventData),
		})
	}

	if params.Key == "" {
		return nil, errx.New("missing key")
	}

	return &params, nil
}
