package emailvalidationsuccess

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/growth/emailvalidator/evs/documents"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"

	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestVerifiedUserOtherNamespace(t *testing.T) {
	mockUsers := &mocks.Logic{}

	mockClients := worker.Clients{
		SQS:   nil,
		Users: mockUsers,
	}
	params := documents.EmailValidationSuccessEventParams{
		Namespace: "SomeNamespace",
		Key:       "SomeKey",
		Email:     "SomeEmail",
	}
	data, err := json.Marshal(params)
	assert.NoError(t, err)

	message := snsmodels.Message{
		Message:           string(data),
		MessageAttributes: map[string]snsmodels.MessageAttribute{"event": snsmodels.MessageAttribute{Type: "String", Value: "Test Event"}},
		Timestamp:         time.Now(),
	}

	ctx := context.Background()
	err = Run(ctx, mockClients, message)
	assert.NoError(t, err)
	mockUsers.AssertNotCalled(t, "SetUserProperties")
}

func TestVerifiedUserUsersServiceNamespace(t *testing.T) {
	ctx := context.Background()
	mockUsers := &mocks.Logic{}
	mockClients := worker.Clients{
		SQS:   nil,
		Users: mockUsers,
	}
	email := "SomeEmail"
	params := documents.EmailValidationSuccessEventParams{
		Namespace: "users-service",
		Key:       "SomeKey",
		Email:     email,
	}
	data, err := json.Marshal(params)
	assert.NoError(t, err)

	userProps := &models.Properties{Email: &email}

	verified := true
	upToUpdate := &models.UpdateableProperties{
		EmailVerified: &verified,
	}

	mockUsers.On("GetUserPropertiesByID", mock.Anything, params.Key, mock.Anything).Return(userProps, nil)
	mockUsers.On("SetUserProperties", mock.Anything, params.Key, upToUpdate).Return(nil)

	message := snsmodels.Message{
		Message:           string(data),
		MessageAttributes: map[string]snsmodels.MessageAttribute{"event": snsmodels.MessageAttribute{Type: "String", Value: "Test Event"}},
		Timestamp:         time.Now(),
	}

	err = Run(ctx, mockClients, message)
	assert.NoError(t, err)
	mockUsers.AssertCalled(t, "SetUserProperties", mock.Anything, params.Key, upToUpdate)
}

func TestVerifiedUserDifferentEmails(t *testing.T) {
	ctx := context.Background()
	mockUsers := &mocks.Logic{}
	mockClients := worker.Clients{
		SQS:   nil,
		Users: mockUsers,
	}
	currEmail := "SomeEmail"
	newEmail := "DifferentEmail"
	params := documents.EmailValidationSuccessEventParams{
		Namespace: "users-service",
		Key:       "SomeKey",
		Email:     newEmail,
	}
	data, err := json.Marshal(params)
	assert.NoError(t, err)

	userProps := &models.Properties{Email: &currEmail}

	verified := false
	upToUpdate := &models.UpdateableProperties{
		EmailVerified: &verified,
	}

	mockUsers.On("GetUserPropertiesByID", mock.Anything, params.Key, mock.Anything).Return(userProps, nil)
	mockUsers.On("SetUserProperties", mock.Anything, params.Key, upToUpdate).Return(nil)

	message := snsmodels.Message{
		Message:           string(data),
		MessageAttributes: map[string]snsmodels.MessageAttribute{"event": snsmodels.MessageAttribute{Type: "String", Value: "Test Event"}},
		Timestamp:         time.Now(),
	}

	err = Run(ctx, mockClients, message)
	assert.NoError(t, err)
	mockUsers.AssertCalled(t, "SetUserProperties", mock.Anything, params.Key, upToUpdate)
}
