package worker

import (
	"encoding/json"
	"fmt"
	"log"
	"sync"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/workerqueue"
	"code.justin.tv/web/users-service/internal/errlog"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// EventParams is the payload containing event data that a TaskFn processes.
type EventParams struct {
	Data      []byte
	Timestamp time.Time
	EventName string
}

type QueueConsumerParams struct {
	NumWorkers           int             `validate:"min=1"`
	QueueName            string          `validate:"nonzero"`
	AccountID            string          // this can by empty
	Done                 <-chan struct{} // validation for this is unsupported
	HandlerFn            EventHandler    // validation for this is unsupported
	KeepTasksHidden      bool            `validate:"nonzero"`
	MaxVisibilityTimeout time.Duration   `validate:"nonzero"`
	LogEventName         string          `validate:"nonzero"`
}

// eventHandler defines a function to handle one type of event.
type EventHandler func(ctx context.Context, clients Clients, params snsmodels.Message) error

// handleSQSMessageFn returns a TaskFn that processes jobs from the users-service queue.
// It is responsible for parsing the SNS -> SQS message, reading the event,
// and calling the given EventHandler with the SQS message body.
func handleSQSMessageFn(parentCtx context.Context, clients Clients, handlerFn EventHandler) workerqueue.TaskFn {
	return func(sqsMsg *sqs.Message) error {
		if sqsMsg == nil {
			return nil
		}

		// Create a child context so we can cancel the parent to terminate all children.
		ctx, cancel := context.WithCancel(parentCtx)
		defer cancel()
		var msg snsmodels.Message
		if err := json.Unmarshal([]byte(*sqsMsg.Body), &msg); err != nil {
			logx.Error(ctx, fmt.Errorf("unmarshaling SNS message: %v", err))
			return errorMessageBodyHandler(ctx, clients, msg)

		}

		return handlerFn(ctx, clients, msg)
	}
}

func CreateConsumers(ctx context.Context, clients Clients, params QueueConsumerParams) (*sync.WaitGroup, error) {
	ctx = logx.WithFields(ctx, logx.Fields{"worker": params.LogEventName})

	// Create workers for queue consumer event.
	log.Printf("spawning %d %s workers listening to queue %s", params.NumWorkers, params.LogEventName, params.QueueName)

	workersParams := workerqueue.CreateWorkersParams{
		AccountID:  params.AccountID,
		NumWorkers: params.NumWorkers,
		QueueName:  params.QueueName,
		Client:     clients.SQS,
		Tasks: map[workerqueue.TaskVersion]workerqueue.TaskFn{
			workerqueue.FallbackTaskVersion: handleSQSMessageFn(ctx, clients, params.HandlerFn),
		},
		KeepTaskHidden:         params.KeepTasksHidden,
		MaxVisibilityTimeout:   params.MaxVisibilityTimeout,
		MessageRetentionPeriod: 60 * 60,
		VisibilityTimeout:      60 * 60,
	}

	wg, errCh, err := workerqueue.CreateWorkers(workersParams, params.Done, clients.Stats)
	if err != nil {
		logx.Error(ctx, err, logx.Fields{
			"queue": params.QueueName,
		})
		return nil, err
	}

	go errlog.FromChannel(ctx, errCh)

	return wg, nil
}

// A placeholder method to avoid loop in worker handler when message body is completely not able to parse
func errorMessageBodyHandler(ctx context.Context, clients Clients, params snsmodels.Message) error {
	return nil
}
