package expirecachequeue

import (
	"encoding/json"
	"fmt"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"code.justin.tv/web/users-service/models"
)

const (
	// EventName is the name of the event which this handler handles.
	EventName = "expire_cache"
)

// Run is the function which should be called to handle the cache expiration event requests
func Run(ctx context.Context, c worker.Clients, msg snsmodels.Message) (err error) {
	defer func() {
		if r := recover(); r != nil {
			err = errx.New(fmt.Sprintf("expirecachequeue.Run circuit panic=%v", r))
		}
	}()

	eventAttr, ok := msg.MessageAttributes["event"]
	if !ok || eventAttr.Type != "String" {
		return errx.New("missing or malformed event", errx.Fields{"event_attr": eventAttr})
	}
	event := eventAttr.Value

	ctx = logx.WithFields(ctx, logx.Fields{"users_service_event": event})

	eventParams := worker.EventParams{
		Data:      []byte(msg.Message),
		Timestamp: msg.Timestamp,
		EventName: event,
	}

	params, err := fetchParams(ctx, eventParams.Data)
	if err != nil {
		return err
	}

	cacheKeys := strings.Split(params.Keys, ",")

	if c.LocalExpirationCacher == nil {
		return errx.New("LocalExpirationCacher is nil")
	}
	return c.LocalExpirationCacher.ExpireLocalCaches(ctx, cacheKeys...)
}

func fetchParams(ctx context.Context, eventData []byte) (*models.SNSExpireCacheEvent, error) {
	var params models.SNSExpireCacheEvent
	if err := json.Unmarshal(eventData, &params); err != nil {
		return nil, errx.New(fmt.Sprintf("Unable to convert json to object for event body: %v", err), errx.Fields{
			"event_body:": string(eventData),
		})
	}

	if params.Keys == "" {
		return nil, errx.New("missing cache keys")
	}

	return &params, nil
}
