package imageuploadqueue

import (
	"encoding/json"

	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/yimg"
	uploadermodel "code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	rpcuploader "code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/users-service/backend"
	uploaderclient "code.justin.tv/web/users-service/internal/clients/uploader"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	Success            = "SUCCESS"
	UserServiceFailure = "BACKEND_FAILURE"
	EventType          = "user_image_update"
)

func Run(ctx context.Context, c worker.Clients, msg snsmodels.Message) (err error) {
	defer func() {
		if r := recover(); r != nil {
			err = errx.New(fmt.Sprintf("imageuploadqueue.Run circuit panic=%v", r))
		}
	}()

	bgCtx := backend.DetachContext(ctx)
	param, err := FetchParam(ctx, msg)
	if err != nil {
		logx.Error(ctx, err)
		return err
	}

	data, image, err := handleUploadImages(ctx, c, param)
	if err != nil {
		go func() {
			pubErr := c.PubSub.Publish(bgCtx, []string{generatePubSubTopic(data.UserID)}, models.UploadImageEvent{
				UserID:    data.UserID,
				ImageType: data.ImageType,
				Status:    UserServiceFailure,
				UploadID:  param.UploadID,
				Type:      EventType,
			})
			if pubErr != nil && !utils.IsHystrixErr(pubErr) {
				logx.Error(bgCtx, pubErr)
			}

			uploaderErr := c.Uploader.UpdateStatus(bgCtx, param.UploadID, rpcuploader.Status_FEATURE_SERVICE_FAILED)
			if uploaderErr != nil && !utils.IsHystrixErr(uploaderErr) {
				logx.Error(bgCtx, uploaderErr)
			}
		}()

		if !utils.IsHystrixErr(err) {
			logx.Error(ctx, err)
		}

		return err
	}

	if image != nil {
		go func() {
			pubErr := c.PubSub.Publish(bgCtx, []string{generatePubSubTopic(data.UserID)}, models.UploadImageEvent{
				UserID:    data.UserID,
				ImageType: data.ImageType,
				NewImage:  *image,
				Status:    Success,
				UploadID:  param.UploadID,
				Type:      EventType,
			})
			if pubErr != nil && !utils.IsHystrixErr(pubErr) {
				logx.Error(bgCtx, pubErr)
			}

			uploaderErr := c.Uploader.UpdateStatus(bgCtx, param.UploadID, rpcuploader.Status_COMPLETE)
			if uploaderErr != nil && !utils.IsHystrixErr(uploaderErr) {
				logx.Error(bgCtx, uploaderErr)
			}

			if metricErr := config.ClusterStatsd().Inc("user_image_upload.success", 1, 0.1); metricErr != nil {
				logx.Warn(bgCtx, fmt.Sprintf("error incrementing user_image_upload.success: %s", metricErr))
			}
		}()
	}

	return nil
}

func handleUploadImages(ctx context.Context, c worker.Clients, param uploadermodel.SNSCallback) (uploaderclient.UserImageMetadata, *yimg.Images, error) {
	bgCtx := backend.DetachContext(ctx)
	userData := uploaderclient.UserImageMetadata{}
	data := param.Data
	if err := json.Unmarshal(data, &userData); err != nil {
		return userData, nil, err
	}

	errorCode := uploader.Status(param.Status)
	if errorCode != uploader.Status_POSTPROCESS_COMPLETE {
		pubErr := c.PubSub.Publish(ctx, []string{generatePubSubTopic(userData.UserID)}, models.UploadImageEvent{
			UserID:    userData.UserID,
			ImageType: userData.ImageType,
			Status:    errorCode.String(),
			UploadID:  param.UploadID,
			Type:      EventType,
		})
		if pubErr != nil && !utils.IsHystrixErr(pubErr) {
			logx.Error(ctx, pubErr)
		}
		return userData, nil, nil
	}

	images, err := c.Users.GetUserImagesByID(ctx, userData.UserID)
	if err != nil {
		logx.Error(ctx, err)
	}

	updatableImageProp := buildUploadImages(ctx, param, userData)
	err = c.Users.SetUserImageProperties(ctx, &updatableImageProp)
	if err != nil {
		return userData, nil, err
	}

	go func() {
		deleteImages(bgCtx, c, userData.UserID, userData.ImageType, images)
	}()

	updatedImages, err := c.Users.GetUserImagesByID(ctx, userData.UserID)
	if err != nil {
		logx.Error(ctx, err)
	}

	var newImage *yimg.Images
	switch userData.ImageType {
	case models.ProfilePictureImageType:
		newImage = updatedImages.ProfileImage
	case models.ProfileBannerImageType:
		newImage = updatedImages.ProfileBanner
	case models.ChannelOfflineImageType:
		newImage = updatedImages.ChannelOfflineImage
	}

	return userData, newImage, nil
}
