package imageuploadqueue_test

import (
	"context"
	"testing"

	"time"

	"code.justin.tv/common/yimg"
	rpcuploader "code.justin.tv/web/upload-service/rpc/uploader"
	pubsubmock "code.justin.tv/web/users-service/internal/clients/pubsub/mocks"
	s3mock "code.justin.tv/web/users-service/internal/clients/s3/mocks"
	uploadermock "code.justin.tv/web/users-service/internal/clients/uploader/mocks"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/internal/worker/imageuploadqueue"
	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	logicmock "code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type UploadWorkerTest struct {
	suite.Suite
	msg      snsmodels.Message
	c        worker.Clients
	users    *logicmock.Logic
	uploader *uploadermock.Uploader
	s3       *s3mock.S3Client
	topics   map[string]string
	pubsub   *pubsubmock.PubSub
}

func (suite *UploadWorkerTest) SetupTest() {
	suite.msg = snsmodels.Message{}
	suite.users = &logicmock.Logic{}
	suite.uploader = &uploadermock.Uploader{}
	suite.s3 = &s3mock.S3Client{}
	suite.pubsub = &pubsubmock.PubSub{}
	suite.topics = map[string]string{worker.ImageUploadQueue: "imagequeue", worker.EmailValidationQueue: "emailqueue"}
	suite.c = worker.Clients{
		Users:    suite.users,
		Uploader: suite.uploader,
		S3:       suite.s3,
		Topics:   suite.topics,
		PubSub:   suite.pubsub,
	}
}

func (suite *UploadWorkerTest) TestSuccessUploadWorkerRun() {
	suite.msg.TopicARN = "imagequeue"
	suite.msg.Message = "{\"upload_id\":\"84acfdfb-e2e8-4a88-b9ed-d466b9f7ef78\",\"output_paths\":[\"s3://ttv-user-pictures-prod/84acfdfb-e2e8-4a88-b9ed-d466b9f7ef78-profile_image-150x150.jpeg\"],\"Data\":\"eyJ1c2VyX2lkIjoiMTM1MDUxNzg2IiwiaW1hZ2VfdHlwZSI6InByb2ZpbGVfaW1hZ2UifQ==\",\"status\":2}"
	uploadedImages := yimg.Images{}
	for _, size := range []string{"150x150"} {
		uploadedImages[size] = yimg.Image{
			Uid:       "84acfdfb-e2e8-4a88-b9ed-d466b9f7ef78",
			Size:      size,
			NewFormat: true,
			Format:    "jpeg",
		}
	}

	images := &models.ImageProperties{
		ID:           "135051786",
		ProfileImage: &uploadedImages,
	}

	paths := []string{"84acfdfb-e2e8-4a88-b9ed-d466b9f7ef78-profile_image-150x150.jpeg"}

	suite.users.On("SetUserImageProperties", mock.Anything, mock.Anything).Return(nil)
	suite.users.On("GetUserImagesByID", mock.Anything, "135051786").Return(images, nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, "135051786", mock.Anything).Return(&models.Properties{ID: "135051786", Login: &images.ID}, nil)
	suite.uploader.On("UpdateStatus", mock.Anything, "84acfdfb-e2e8-4a88-b9ed-d466b9f7ef78", rpcuploader.Status_COMPLETE).Return(nil)
	suite.s3.On("BatchDelete", paths).Return(nil)
	suite.pubsub.On("Publish", mock.Anything, []string{"user-image-update.135051786"}, mock.Anything, mock.Anything).Return(nil)

	err := imageuploadqueue.Run(context.Background(), suite.c, suite.msg)
	time.Sleep(2 * time.Second)

	assert.NoError(suite.T(), err)
	suite.users.AssertExpectations(suite.T())
	suite.s3.AssertExpectations(suite.T())
	suite.uploader.AssertExpectations(suite.T())
}

func TestUploadWorkerSuite(t *testing.T) {
	suite.Run(t, new(UploadWorkerTest))
}
