package imageuploadqueue

import (
	"encoding/json"

	"code.justin.tv/web/users-service/internal/worker/snsmodels"
	"golang.org/x/net/context"

	"fmt"
	"strconv"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/yimg"
	uploader "code.justin.tv/web/upload-service/models"
	uploaderclient "code.justin.tv/web/users-service/internal/clients/uploader"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/internal/worker"
	"code.justin.tv/web/users-service/models"
)

func FetchParam(ctx context.Context, msg snsmodels.Message) (uploader.SNSCallback, error) {
	var param uploader.SNSCallback
	if err := json.Unmarshal([]byte(msg.Message), &param); err != nil {
		return param, err
	}
	return param, nil
}

func buildUploadImages(ctx context.Context, param uploader.SNSCallback, data uploaderclient.UserImageMetadata) models.ImageProperties {
	id := data.UserID
	imageType := data.ImageType
	imageProp := models.ImageProperties{}
	if len(param.Outputs) == 0 {
		return imageProp
	}
	format := param.Outputs[0].Format
	uploadedImages := yimg.Images{}

	for _, outputPath := range param.Outputs {
		if imageType == models.ProfilePictureImageType {
			uploadedImages[outputPath.Dimensions] = yimg.Image{
				Uid:       param.UploadID,
				Size:      outputPath.Dimensions,
				NewFormat: true,
				Format:    format,
			}
		} else if imageType == models.ChannelOfflineImageType {
			uploadedImages[yimg.ChannelOfflineImageSizes[0]] = yimg.Image{
				Uid:       param.UploadID,
				Size:      yimg.ChannelOfflineImageSizes[0],
				NewFormat: true,
				Format:    format,
			}
		} else if imageType == models.ProfileBannerImageType {
			height, err := strconv.ParseInt(yimg.ProfileBannerSizes[0], 10, 64)
			if err != nil {
				logx.Error(ctx, err)
			}
			uploadedImages[yimg.ProfileBannerSizes[0]] = yimg.Image{
				Uid:       param.UploadID,
				Height:    int(height),
				NewFormat: true,
				Format:    format,
			}
		}
	}

	imageProp.ID = id
	switch imageType {
	case models.ProfileBannerImageType:
		imageProp.ProfileBanner = &uploadedImages
	case models.ProfilePictureImageType:
		imageProp.ProfileImage = &uploadedImages
	case models.ChannelOfflineImageType:
		imageProp.ChannelOfflineImage = &uploadedImages
	}

	return imageProp
}

func deleteImages(ctx context.Context, c worker.Clients, id, imageType string, images *models.ImageProperties) {
	user, err := c.Users.GetUserPropertiesByID(ctx, id, nil)
	if user == nil || err != nil {
		logx.Error(ctx, errx.New("Can not find user for image deletion"))
		return
	}

	paths := []string{}
	var f models.ImageFunc
	var imagesToDelete *yimg.Images

	switch imageType {
	case models.ProfilePictureImageType:
		if images.DefaultProfileImage == nil {
			imagesToDelete = images.ProfileImage
			f = yimg.ProfileImagesName
		}
	case models.ChannelOfflineImageType:
		if images.DefaultChannelOfflineImage == nil {
			imagesToDelete = images.ChannelOfflineImage
			f = yimg.ChannelImageName
		}
	case models.ProfileBannerImageType:
		if images.DefaultProfileBanner == nil {
			imagesToDelete = images.ProfileBanner
			f = yimg.ProfileBannersName
		}
	}

	if imagesToDelete != nil {
		for _, image := range *imagesToDelete {
			path, e := f(image, *user.Login)
			if e == nil {
				paths = append(paths, path)
			} else if !utils.IsHystrixErr(e) {
				logx.Warn(ctx, e, logx.Fields{
					"login": *user.Login,
					"image": image,
					"error": e.Error(),
				})
			}
		}

		if len(paths) != 0 {
			err = c.S3.BatchDelete(paths)
			if err != nil && !utils.IsHystrixErr(err) {
				pathBytes, e := json.Marshal(paths)
				if e != nil {
					logx.Error(ctx, e)
				}
				logx.Error(ctx, err, logx.Fields{
					"paths":      string(pathBytes),
					"id":         id,
					"image_type": imageType,
				})
			}
		}
	}
}

func generatePubSubTopic(id string) string {
	return fmt.Sprintf("user-image-update.%s", id)
}
