package imageuploadqueue

import (
	"context"
	"testing"

	"code.justin.tv/common/yimg"
	uploader "code.justin.tv/web/upload-service/models"
	uploaderclient "code.justin.tv/web/users-service/internal/clients/uploader"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
)

func TestBuildUploadImages(t *testing.T) {
	uploadID := "9956c277-8992-4a29-9402-6cfaa547bdc8"
	output600 := uploader.OutputInfo{
		Path:         "s3://ttv-user-pictures-prod/9956c277-8992-4a29-9402-6cfaa547bdc8-profile_image-600x600.jpg",
		Name:         "9956c277-8992-4a29-9402-6cfaa547bdc8-profile_image-600x600.jpg",
		NameTemplate: "{{UploadID}}-profile_image-{{Dimensions}}{{Extension}}",
		Format:       "jpg",
		Width:        600,
		Height:       600,
		Dimensions:   "600x600",
	}

	output300 := uploader.OutputInfo{
		Path:         "s3://ttv-user-pictures-prod/9956c277-8992-4a29-9402-6cfaa547bdc8-profile_image-300x300.jpg",
		Name:         "9956c277-8992-4a29-9402-6cfaa547bdc8-profile_image-300x300.jpg",
		NameTemplate: "{{UploadID}}-profile_image-{{Dimensions}}{{Extension}}",
		Format:       "jpg",
		Width:        300,
		Height:       300,
		Dimensions:   "300x300",
	}

	data := "eyJ1c2VyX2lkIjoiMTMxMTgwOTEzIiwiaW1hZ2VfdHlwZSI6InByb2ZpbGVfaW1hZ2UiLCJmb3JtYXQiOiJqcGcifQ=="
	callback := uploader.SNSCallback{
		Status:   2,
		UploadID: uploadID,
		Data:     []byte(data),
		Outputs:  []uploader.OutputInfo{output300, output600},
	}

	image600 := yimg.Image{
		Uid:       "9956c277-8992-4a29-9402-6cfaa547bdc8",
		Size:      "600x600",
		Format:    "jpg",
		NewFormat: true,
	}

	image300 := yimg.Image{
		Uid:       "9956c277-8992-4a29-9402-6cfaa547bdc8",
		Size:      "300x300",
		Format:    "jpg",
		NewFormat: true,
	}

	bannerImage := yimg.Image{
		Uid:       "9956c277-8992-4a29-9402-6cfaa547bdc8",
		Height:    480,
		Format:    "jpg",
		NewFormat: true,
	}

	offlineImage := yimg.Image{
		Uid:       "9956c277-8992-4a29-9402-6cfaa547bdc8",
		Size:      "1920x1080",
		Format:    "jpg",
		NewFormat: true,
	}

	images := models.ImageProperties{
		ID:           "131180913",
		ProfileImage: &yimg.Images{"600x600": image600, "300x300": image300},
	}

	bannerImages := models.ImageProperties{
		ID:            "131180913",
		ProfileBanner: &yimg.Images{"480": bannerImage},
	}

	offlineImages := models.ImageProperties{
		ID:                  "131180913",
		ChannelOfflineImage: &yimg.Images{"1920x1080": offlineImage},
	}

	for _, scenario := range []struct {
		Name      string
		Images    models.ImageProperties
		ImageType string
		Callback  uploader.SNSCallback
	}{
		{
			Name:      "build profile image",
			Callback:  callback,
			Images:    images,
			ImageType: models.ProfilePictureImageType,
		},
		{
			Name:      "build profile banner",
			Callback:  callback,
			Images:    bannerImages,
			ImageType: models.ProfileBannerImageType,
		},
		{
			Name:      "build channel offline image",
			Callback:  callback,
			Images:    offlineImages,
			ImageType: models.ChannelOfflineImageType,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			userData := uploaderclient.UserImageMetadata{
				UserID:    "131180913",
				ImageType: scenario.ImageType,
				Format:    "jpg",
			}
			images := buildUploadImages(context.Background(), scenario.Callback, userData)
			assert.Equal(t, scenario.Images.ID, images.ID)

			switch scenario.ImageType {
			case models.ProfilePictureImageType:
				assert.Equal(t, *scenario.Images.ProfileImage, *images.ProfileImage)
				assert.Nil(t, scenario.Images.ChannelOfflineImage)
				assert.Nil(t, scenario.Images.ProfileBanner)
			case models.ProfileBannerImageType:
				assert.Equal(t, *scenario.Images.ProfileBanner, *images.ProfileBanner)
				assert.Nil(t, scenario.Images.ChannelOfflineImage)
				assert.Nil(t, scenario.Images.ProfileImage)
			default:
				assert.Equal(t, *scenario.Images.ChannelOfflineImage, *images.ChannelOfflineImage)
				assert.Nil(t, scenario.Images.ProfileBanner)
				assert.Nil(t, scenario.Images.ProfileImage)
			}
		})
	}
}
