package logic

import (
	"strconv"

	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	leviathan "code.justin.tv/chat/leviathan/client"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/internal/clients/hystrix"
	"code.justin.tv/web/users-service/models"
	"github.com/pkg/errors"
	"golang.org/x/net/context"
)

const (
	origin = "users-service"
)

func (l *logicImpl) BanUser(ctx context.Context, banProp *models.BanUserProperties) error {
	var strikes int

	if banProp.Permanent {
		strikes = 3
	} else {
		strikes = 1
	}

	up, err := l.users.GetUserPropertiesByID(ctx, banProp.TargetUserID, util.ReadOptions{ReadFromMaster: true})
	if err != nil {
		return err
	}

	var dmcaStrikes, tosStrikes int
	if up.DmcaViolationCount != nil {
		dmcaStrikes = *up.DmcaViolationCount
	}

	if up.TosViolationCount != nil {
		tosStrikes = *up.TosViolationCount
	}

	var tosBan bool

	if banProp.Type == "dmca" {
		dmcaStrikes += strikes
		tosBan = false
	} else if banProp.Type == "tos" {
		tosStrikes += strikes
		tosBan = true
	} else {
		return errors.New("Wrong type for ban.")
	}

	warn := false
	var duration string
	if dmcaStrikes < 3 && tosStrikes < 3 {
		warn = true
		duration = strconv.Itoa(60 * 60 * 24)
	}

	if up.RemoteIP != nil && *up.RemoteIP != "" {
		if !banProp.SkipIPBan {
			if err = l.blacklist.AddIP(ctx, *up.RemoteIP, duration, nil); err != nil {
				return errx.New(err)
			}
		}
	}

	err = hystrix.Do(hystrix.CommandHideAllFollows, func() error {
		return l.follows.HideAllFollows(ctx, banProp.TargetUserID, nil)
	}, nil)
	if err != nil {
		return errx.New(err)
	}

	var reportReason string
	if banProp.Reason == "" {
		reportReason = "unknown"
	} else {
		reportReason = banProp.Reason
	}
	if err := l.banreport.ReportBan(ctx, banProp.TargetUserID, banProp.FromUserID, banProp.Type, warn, reportReason, nil); err != nil {
		return errx.New(err)
	}

	if err := l.users.BanUser(ctx, banProp.TargetUserID, warn, tosBan, dmcaStrikes, tosStrikes); err != nil {
		return err
	}

	l.publishBan(ctx, duration, tosBan, warn, banProp)

	return nil
}

func (l *logicImpl) publishBan(ctx context.Context, duration string, tosBan, warn bool, banProp *models.BanUserProperties) {
	bgCtx := backend.DetachContext(ctx)

	go func() {
		if err := l.rails.DeleteCache(bgCtx, banProp.TargetUserID, nil); err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	reporter := "admin"
	if banProp.FromUserID != "" {
		reporter = banProp.FromUserID
	}
	go func() {
		banEvent := events.BanUserEvent(reporter, banProp.TargetUserID, banProp.Reason)
		l.auditor.Audit(bgCtx, banEvent)
	}()

	ipDuration := int64(0)
	if !banProp.SkipIPBan {
		id, err := strconv.ParseInt(duration, 10, 64)
		if err != nil {
			id = int64(0)
		}
		ipDuration = id
	}

	banDuration := int64(-1)
	if warn {
		banDuration = int64(60 * 60 * 24)
	}

	go func() {
		if configs.IsProduction() {
			description := banProp.Description

			// For DMCA strikes that do not have a description, default the descripton to be 'DMCA tool'
			if banProp.Type == "dmca" && banProp.Description == "" {
				description = "DMCA tool"
			}
			err := l.leviathan.TosSuspensionReport(bgCtx, leviathan.TosSuspensionReportParam{
				FromUserID:     reporter,
				TargetUserID:   banProp.TargetUserID,
				Reason:         banProp.Reason,
				DetailedReason: banProp.DetailedReason,
				Content:        banProp.Content,
				Description:    description,
				Duration:       banDuration,
				ClearImages:    banProp.ClearImages,
				Origin:         origin,
				IPBan:          ipDuration,
			})
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}
	}()

	// For tos ban, send email notification via pushy
	if banProp.Type == "tos" {
		go func() {
			event := models.SNSBanUserEvent{
				RecipientID: banProp.TargetUserID,
				Type:        banProp.Type,
				Warn:        warn,
				Reason:      banProp.Reason,
				Description: banProp.DetailedReason,
			}
			if err := l.sns.PublishBanUser(bgCtx, event); err != nil {
				logx.Error(bgCtx, err)
			}
		}()
	}

	go func() {
		err := l.spade.TrackEvent(bgCtx, "ban_user", models.BanUserEvent{
			ReporterID:    banProp.FromUserID,
			TargetID:      banProp.TargetUserID,
			Reason:        banProp.Reason,
			Duration:      duration,
			IPBanDuration: ipDuration,
			Type:          banProp.Type,
			Env:           configs.Environment(),
		})
		if err != nil {
			logx.Error(bgCtx, "failed to report spade ban user event", logx.Fields{
				"error": err,
			})
		}
	}()

	go func() {
		event := models.SNSBanEvent{
			UserID:                  banProp.TargetUserID,
			DmcaViolation:           !tosBan,
			TermsOfServiceViolation: tosBan,
			Timestamp:               time.Now(),
		}
		if err := l.sns.PublishBan(bgCtx, event); err != nil {
			logx.Error(bgCtx, err)
		}
	}()
}
