package logic

import (
	"crypto/sha1"
	"fmt"
	"math/rand"
	"strconv"
	"time"

	"strings"

	"golang.org/x/sync/errgroup"

	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
	"golang.org/x/net/context"
)

const (
	Salt                     = "wasiteveraquestion"
	EmailValidationNamespace = "users-service"
	maxAllowed               = 25
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

func (l *logicImpl) CreateUser(ctx context.Context, cup *models.CreateUserProperties) (*models.Properties, error) {
	bgCtx := backend.DetachContext(ctx)

	users, err := l.GetUserPropertiesBulk(ctx, &models.FilterParams{Emails: []string{cup.Email}})
	if err != nil {
		return nil, err
	}

	if len(users) > maxAllowed && !validators.IsWhitelistedEmail(&cup.Email) {
		if metricErr := config.ClusterStatsd().Inc("user_creation.email_with_too_many_users", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing user creation metrics duration: %s", metricErr))
		}
		return nil, models.TooManyUsersAssociatedWithEmail
	}

	// explicitly do lower case conversion for login
	cup.Displayname = cup.Login
	cup.Login = strings.ToLower(cup.Login)
	if err := validators.IsLoginValid(&cup.Login, false); err != nil {
		return nil, err
	}

	var g errgroup.Group
	var rp []models.LoginTypeProperties

	// See if the desired login is reserved
	g.Go(func() error {
		var err error
		rp, _, err = l.GetTypeForLogins(ctx, []string{cup.Login}, "", false, false)
		return err
	})

	xErr := g.Wait()
	// if up isn't nil, user already exists. if rp isn't nil, login is reserved.
	if len(rp) != 0 || xErr != nil {
		return nil, models.ErrLoginNotAvailable
	}

	if cup.IP != "" {
		b, err := l.blacklist.ValidateIP(ctx, cup.IP, nil)
		if err != nil {
			return nil, err
		}
		if b {
			return nil, models.ErrIPBlacklisted
		}

		location, _ := l.geoIP.GetCountry(cup.IP)
		cup.Location = location
	}

	if err := l.users.CreateUser(ctx, cup); err != nil {
		return nil, err
	}

	up, err := l.users.GetUserPropertiesByLogin(ctx, cup.Login, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true})
	if up == nil || err != nil {
		return nil, err
	}
	newuser := *up

	go func() {
		err := l.createAndSendNewUserVerificationEmail(bgCtx, newuser)
		if err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	go func() {
		creationEvents := events.CreateUserEvent(newuser.ID, *newuser.Login)
		l.auditor.Audit(bgCtx, creationEvents)
	}()

	go func() {
		err = l.sns.PublishCreation(bgCtx, models.SNSCreationEvent{
			UserID:      newuser.ID,
			Login:       *newuser.Login,
			Displayname: *newuser.Displayname,
			Timestamp:   time.Now(),
		})
		if err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	go func() {
		err := l.spade.TrackEvent(bgCtx, "signup", models.SignupEvent{
			UserID:   newuser.ID,
			Login:    cup.Login,
			IP:       cup.IP,
			DeviceID: cup.DeviceID,
			Env:      configs.Environment(),
		})
		if err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	return &newuser, nil
}

func (l *logicImpl) createAndSendNewUserVerificationEmail(ctx context.Context, newuser models.Properties) error {
	code, err := generateEmailVerificationCode()
	if err != nil {
		return err
	}

	err = l.users.CreateUserEmail(ctx, newuser.ID, code)
	if err != nil {
		return err
	}

	return l.sendNewUserVerificationEmailToEVS(ctx, newuser)
}

func (l *logicImpl) sendNewUserVerificationEmailToEVS(ctx context.Context, newuser models.Properties) error {
	locale := "en"
	if newuser.Language != nil && *newuser.Language != "" {
		locale = *newuser.Language
	}

	_, err := l.evs.AddVerificationRequest(ctx, EmailValidationNamespace, newuser.ID, *newuser.Email, locale, "authenticating the account for "+*newuser.Login, nil)
	return err
}

func generateEmailVerificationCode() (string, error) {
	hasher := sha1.New()
	s := fmt.Sprintf("%s%s", Salt, strconv.FormatInt(rand.Int63n(1000000), 10))
	_, err := hasher.Write([]byte(s))
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("%x\n", hasher.Sum(nil))[0:9], nil
}
