package logic

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/logx"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

var whitelistedDeleteEnvironments = []string{"development", "test", "staging", "production"}

type hardDeleteUserAllower interface {
	hardDeleteUserAllowed(ctx context.Context, ID string) error
}

func (l *logicImpl) HardDeleteUser(ctx context.Context, ID string, adminLogin string, skipBlock bool) error {
	err := l.deleteAllower.hardDeleteUserAllowed(ctx, ID)
	if err != nil {
		return err
	}

	if err := l.rails.HardDeleteUser(ctx, ID, adminLogin, nil); err != nil {
		return err
	}

	_, err = l.users.HardDeleteUser(ctx, ID, skipBlock)
	if err != nil {
		return err
	}

	if err := l.channels.DeleteChannel(ctx, ID); err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to delete channel: %s", err))
	}

	if err := l.rails.DeleteCache(ctx, ID, nil); err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to delete from rails cache: %s", err))
	}

	bgCtx := backend.DetachContext(ctx)
	// Audit hard delete
	go l.auditor.Audit(bgCtx, events.HardDelete(ID, adminLogin))

	// SNS hard delete
	go func() {
		if err := l.sns.PublishHardDelete(bgCtx, models.SNSHardDeleteEvent{
			UserID:    ID,
			Timestamp: time.Now(),
		}); err != nil {
			logx.Error(bgCtx, fmt.Sprintf("failed to publish hard delete: %s", err))
		}
	}()

	return nil
}

func (l *logicImpl) hardDeleteUserAllowed(ctx context.Context, ID string) error {
	// Environment check
	env := configs.Environment()
	var isWhitelisted bool
	for _, whitelisted := range whitelistedDeleteEnvironments {
		isWhitelisted = whitelisted == env
		if isWhitelisted {
			break
		}
	}

	if !isWhitelisted {
		err := models.ErrNotAllowedToHardDelete
		err.ErrorValue = "Delete must run in whitelisted environment"
		return err
	}

	// Has user been deleted?
	props, err := l.users.GetUserPropertiesBulk(ctx, &models.FilterParams{
		IDs:        []string{ID},
		NotDeleted: false,
	})

	if err != nil {
		return err
	}

	if len(props) != 1 {
		err := models.ErrNotAllowedToHardDelete
		err.ErrorValue = "Unable to find user"
		return err
	}

	user := props[0]

	if user.DeletedOn == nil {
		err := models.ErrNotAllowedToHardDelete
		err.ErrorValue = "You must soft delete prior to hard delete"
		return err
	}

	// Does user belong to team?

	// Does user have oauth2 clients?
	filters := map[owl.FilterableColumn]string{
		owl.FilterColOwnerID: ID,
	}
	clients, _, err := l.owl.GetClients(ctx, "", false, filters, owl.SortKeyClientName, owl.SortAsc, nil)
	if err != nil {
		return err
	}

	if len(clients) > 0 {
		err := models.ErrNotAllowedToHardDelete
		err.ErrorValue = "User owns an oauth2 client which other users have authed with. This case requires manual intervention by an engineer to verify that we are not inadvertently deleting an important oauth2 client."
		return err
	}

	// Is user partner?
	isPartnered, err := l.partners.CheckPartneredByUserID(ctx, ID, nil)
	if err != nil {
		return err
	}

	if isPartnered {
		err := models.ErrNotAllowedToHardDelete
		err.ErrorValue = "User is a partner. Unpartner before hard delete."
		return err
	}

	return nil
}

func (l *logicImpl) SoftDeleteUser(ctx context.Context, ID string, adminLogin string) error {
	if err := l.rails.SoftDeleteUser(ctx, ID, nil); err != nil {
		return err
	}

	if err := l.users.SoftDeleteUser(ctx, ID); err != nil {
		return err
	}

	if err := l.rails.DeleteCache(ctx, ID, nil); err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to delete from rails cache: %s", err))
	}

	if err := l.channels.DeleteChannel(ctx, ID); err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to delete channel: %s", err))
	}

	bgCtx := backend.DetachContext(ctx)
	// Audit soft delete
	go l.auditor.Audit(bgCtx, events.SoftDelete(ID, adminLogin))

	// SNS soft delete
	go func() {
		if err := l.sns.PublishSoftDelete(ctx, models.SNSSoftDeleteEvent{
			UserID:    ID,
			Timestamp: time.Now(),
		}); err != nil {
			logx.Error(bgCtx, fmt.Sprintf("failed to publish soft delete: %s", err))
		}
	}()

	return nil
}

func (l *logicImpl) UndeleteUser(ctx context.Context, ID string, adminLogin string) error {
	if err := l.users.UndeleteUser(ctx, ID); err != nil {
		return err
	}

	if err := l.rails.UndeleteUser(ctx, ID, nil); err != nil {
		return err
	}

	bgCtx := backend.DetachContext(ctx)
	// Audit undelete
	go l.auditor.Audit(bgCtx, events.Undelete(ID, adminLogin))

	// SNS undelete
	go func() {
		if err := l.sns.PublishUndelete(ctx, models.SNSUndeleteEvent{
			UserID:    ID,
			Timestamp: time.Now(),
		}); err != nil {
			logx.Error(bgCtx, fmt.Sprintf("failed to publish undelete: %s", err))
		}
	}()

	return nil
}
