package logic

import (
	"errors"
	"testing"
	"time"

	"golang.org/x/net/context"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/web/owl/oauth2"
	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	usersMocks "code.justin.tv/web/users-service/backend/users/mocks"
	auditorMocks "code.justin.tv/web/users-service/internal/clients/auditor/mocks"
	owlMocks "code.justin.tv/web/users-service/internal/clients/owl/mocks"
	partnerMocks "code.justin.tv/web/users-service/internal/clients/partnerships/mocks"
	railsMocks "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	snsMocks "code.justin.tv/web/users-service/internal/clients/sns/mocks"
	"code.justin.tv/web/users-service/models"
)

func mockGetUserSuccess(userMock *usersMocks.Backend) {
	now := time.Now()
	userMock.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return([]models.Properties{{
		DeletedOn: &now,
	}}, nil)
}

func mockGetOwlClientSuccess(owlMock *owlMocks.Client) {
	owlMock.On("GetClients", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, "", nil)
}

func TestHardDeleteUserAllowed(t *testing.T) {
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, l *logicImpl)
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, l *logicImpl) {
				userMock := &usersMocks.Backend{}
				mockGetUserSuccess(userMock)
				l.users = userMock

				owlMock := &owlMocks.Client{}
				mockGetOwlClientSuccess(owlMock)
				l.owl = owlMock

				partnersMock := &partnerMocks.Client{}
				partnersMock.On("CheckPartneredByUserID", mock.Anything, mock.Anything, mock.Anything).Return(false, nil)
				l.partners = partnersMock
			},
			Error: nil,
		},
		{
			Name: "soft delete before hard delete",
			Setup: func(t *testing.T, l *logicImpl) {
				userMock := &usersMocks.Backend{}

				userMock.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return([]models.Properties{{}}, nil)
				l.users = userMock
			},
			Error: models.ErrNotAllowedToHardDelete,
		},
		{
			Name: "user not found",
			Setup: func(t *testing.T, l *logicImpl) {
				userMock := &usersMocks.Backend{}

				userMock.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return(nil, nil)
				l.users = userMock
			},
			Error: models.ErrNotAllowedToHardDelete,
		},
		{
			Name: "user has oauth2 clients",
			Setup: func(t *testing.T, l *logicImpl) {
				userMock := &usersMocks.Backend{}
				mockGetUserSuccess(userMock)
				l.users = userMock

				owlMock := &owlMocks.Client{}
				owlMock.On("GetClients", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]*oauth2.Client{{}}, "", nil)
				l.owl = owlMock
			},
			Error: models.ErrNotAllowedToHardDelete,
		},
		{
			Name: "user is partnered",
			Setup: func(t *testing.T, l *logicImpl) {
				userMock := &usersMocks.Backend{}
				mockGetUserSuccess(userMock)
				l.users = userMock

				owlMock := &owlMocks.Client{}
				mockGetOwlClientSuccess(owlMock)
				l.owl = owlMock

				partnersMock := &partnerMocks.Client{}
				partnersMock.On("CheckPartneredByUserID", mock.Anything, mock.Anything, mock.Anything).Return(true, nil)
				l.partners = partnersMock
			},
			Error: models.ErrNotAllowedToHardDelete,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.hardDeleteUserAllowed(context.Background(), "1")
			if err != scenario.Error {
				t.Fatalf("errors do not match: %v vs %v", err, scenario.Error)
			}
		})
	}
}

func TestHardDeleteUser(t *testing.T) {
	for _, scenario := range []struct {
		Name      string
		ID, Admin string
		SkipBlock bool
		Setup     func(t *testing.T, l *logicImpl)
		Failure   bool
	}{
		{
			Name:      "hard delete not allowed",
			ID:        "12",
			Admin:     "test",
			SkipBlock: false,
			Failure:   true,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: false}
			},
		},
		{
			Name:      "rails hard delete fails",
			ID:        "12",
			Admin:     "test",
			SkipBlock: false,
			Failure:   true,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("HardDeleteUser", mock.Anything, "12", "test", mock.Anything).Return(errors.New("failure"))
				l.rails = railsMock
			},
		},
		{
			Name:      "users hard delete fails",
			ID:        "12",
			Admin:     "test",
			SkipBlock: false,
			Failure:   true,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("HardDeleteUser", mock.Anything, "12", "test", mock.Anything).Return(nil)
				l.rails = railsMock

				usersMock := &usersMocks.Backend{}
				usersMock.On("HardDeleteUser", mock.Anything, "12", false).Return(nil, errors.New("failure"))
				l.users = usersMock
			},
		},
		{
			Name:      "happy path",
			ID:        "12",
			Admin:     "test",
			SkipBlock: false,
			Failure:   false,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("HardDeleteUser", mock.Anything, "12", "test", mock.Anything).Return(nil)
				railsMock.On("DeleteCache", mock.Anything, "12", mock.Anything).Return(nil)
				l.rails = railsMock

				usersMock := &usersMocks.Backend{}
				usersMock.On("HardDeleteUser", mock.Anything, "12", false).Return(nil, nil)
				l.users = usersMock

				channelsMock := &channelsMocks.Backend{}
				channelsMock.On("DeleteChannel", mock.Anything, "12").Return(nil)
				l.channels = channelsMock

				auditorMock := &auditorMocks.Auditor{}
				auditorMock.On("Audit", mock.Anything, mock.Anything)
				l.auditor = auditorMock

				snsMock := &snsMocks.Publisher{}
				snsMock.On("PublishHardDelete", mock.Anything, mock.Anything).Return(nil)
				l.sns = snsMock
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.HardDeleteUser(context.Background(), scenario.ID, scenario.Admin, scenario.SkipBlock)
			if err != nil && !scenario.Failure {
				t.Fatal("expected failure but hard delete didn't fail")
			}
			if err == nil && scenario.Failure {
				t.Fatal("expected success but hard delete failed:", err)
			}
		})
	}
}

type mockHardDeleteUserAllower struct {
	allow bool
}

func (m *mockHardDeleteUserAllower) hardDeleteUserAllowed(ctx context.Context, ID string) error {
	if m.allow {
		return nil
	}

	return errors.New("not allowed")
}

func TestSoftDeleteUser(t *testing.T) {
	for _, scenario := range []struct {
		Name      string
		ID, Admin string
		Setup     func(t *testing.T, l *logicImpl)
		Failure   bool
	}{
		{
			Name:    "rails soft delete fails",
			ID:      "12",
			Admin:   "test",
			Failure: true,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("SoftDeleteUser", mock.Anything, "12", mock.Anything).Return(errors.New("failure"))
				l.rails = railsMock
			},
		},
		{
			Name:    "users soft delete fails",
			ID:      "12",
			Admin:   "test",
			Failure: true,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("SoftDeleteUser", mock.Anything, "12", mock.Anything).Return(nil)
				l.rails = railsMock

				usersMock := &usersMocks.Backend{}
				usersMock.On("SoftDeleteUser", mock.Anything, "12").Return(errors.New("failure"))
				l.users = usersMock
			},
		},
		{
			Name:    "happy path",
			ID:      "12",
			Admin:   "test",
			Failure: false,
			Setup: func(t *testing.T, l *logicImpl) {
				l.deleteAllower = &mockHardDeleteUserAllower{allow: true}

				railsMock := &railsMocks.Client{}
				railsMock.On("SoftDeleteUser", mock.Anything, "12", mock.Anything).Return(nil)
				railsMock.On("DeleteCache", mock.Anything, "12", mock.Anything).Return(nil)
				l.rails = railsMock

				usersMock := &usersMocks.Backend{}
				usersMock.On("SoftDeleteUser", mock.Anything, "12").Return(nil)
				l.users = usersMock

				channelsMock := &channelsMocks.Backend{}
				channelsMock.On("DeleteChannel", mock.Anything, "12").Return(nil)
				l.channels = channelsMock

				auditorMock := &auditorMocks.Auditor{}
				auditorMock.On("Audit", mock.Anything, mock.Anything)
				l.auditor = auditorMock

				snsMock := &snsMocks.Publisher{}
				snsMock.On("PublishSoftDelete", mock.Anything, mock.Anything).Return(nil)
				l.sns = snsMock
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.SoftDeleteUser(context.Background(), scenario.ID, scenario.Admin)
			if err != nil && !scenario.Failure {
				t.Fatal("expected failure but soft delete didn't fail")
			}
			if err == nil && scenario.Failure {
				t.Fatal("expected success but soft delete failed:", err)
			}
		})
	}
}

func TestUndeleteUser(t *testing.T) {

	for _, scenario := range []struct {
		Name      string
		ID, Admin string
		Setup     func(t *testing.T, l *logicImpl)
		Failure   bool
	}{
		{
			Name:    "happy path",
			ID:      "12",
			Admin:   "test",
			Failure: false,
			Setup: func(t *testing.T, l *logicImpl) {
				usersMock := &usersMocks.Backend{}
				usersMock.On("UndeleteUser", mock.Anything, "12").Return(nil)
				l.users = usersMock

				railsMock := &railsMocks.Client{}
				railsMock.On("UndeleteUser", mock.Anything, "12", mock.Anything).Return(nil)
				l.rails = railsMock

				auditorMock := &auditorMocks.Auditor{}
				auditorMock.On("Audit", mock.Anything, mock.Anything)
				l.auditor = auditorMock

				snsMock := &snsMocks.Publisher{}
				snsMock.On("PublishUndelete", mock.Anything, mock.Anything).Return(nil)
				l.sns = snsMock
			},
		}, {
			Name:    "user undelete fails",
			ID:      "12",
			Admin:   "test",
			Failure: true,
			Setup: func(t *testing.T, l *logicImpl) {
				usersMock := &usersMocks.Backend{}
				usersMock.On("UndeleteUser", mock.Anything, "12").Return(errors.New("failure"))
				l.users = usersMock
			},
		}, {
			Name:    "rails undelete fails",
			ID:      "12",
			Admin:   "test",
			Failure: true,
			Setup: func(t *testing.T, l *logicImpl) {
				usersMock := &usersMocks.Backend{}
				usersMock.On("UndeleteUser", mock.Anything, "12").Return(nil)
				l.users = usersMock

				railsMock := &railsMocks.Client{}
				railsMock.On("UndeleteUser", mock.Anything, "12", mock.Anything).Return(errors.New("failure"))
				l.rails = railsMock
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.UndeleteUser(context.Background(), scenario.ID, scenario.Admin)
			if err != nil && !scenario.Failure {
				t.Fatal("expected failure but undelete didn't fail")
			}

			if err == nil && scenario.Failure {
				t.Fatal("expected success but undelete failed:", err)
			}
		})
	}
}
