package logic

import (
	"strconv"
	"time"

	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	leviathan "code.justin.tv/chat/leviathan/client"
	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/feeds/following-service/client/follows"
	evs "code.justin.tv/growth/emailvalidator/evs/client"
	"code.justin.tv/web/blacklist"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/channels"
	"code.justin.tv/web/users-service/backend/reservations"
	"code.justin.tv/web/users-service/backend/users"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/banreport"
	discovery "code.justin.tv/web/users-service/internal/clients/discovery"
	"code.justin.tv/web/users-service/internal/clients/geoip"
	"code.justin.tv/web/users-service/internal/clients/kinesis"
	"code.justin.tv/web/users-service/internal/clients/partnerships"
	"code.justin.tv/web/users-service/internal/clients/rails"
	"code.justin.tv/web/users-service/internal/clients/s3"
	"code.justin.tv/web/users-service/internal/clients/sns"
	"code.justin.tv/web/users-service/internal/clients/twilio"
	"code.justin.tv/web/users-service/internal/clients/uploader"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
)

//go:generate mockery -name Logic
type Logic interface {
	GetUserPropertiesByID(ctx context.Context, ID string, params *models.FilterParams) (*models.Properties, error)
	GetUserPropertiesBulk(ctx context.Context, params *models.FilterParams) ([]models.Properties, error)
	GetUserPropertiesLike(ctx context.Context, field string, pattern string) ([]models.Properties, error)
	GetTypeForLogins(ctx context.Context, logins []string, requestingUserID string, isRename, overrideLoginLength bool) ([]models.LoginTypeProperties, bool, error)
	GetBannedUsers(ctx context.Context, until time.Time) ([]models.Properties, error)
	GetRenameEligibility(ctx context.Context, ID string) (*models.RenameProperties, error)
	SetUserProperties(ctx context.Context, ID string, up *models.UpdateableProperties) error
	SetUserImageProperties(ctx context.Context, updatableImageProp *models.ImageProperties) error
	UploadUserImages(ctx context.Context, updatableImageProp *models.UploadableImage) (models.UploadInfo, error)
	UnbanUser(ctx context.Context, ID string, ifResetCount bool) error
	BanUser(ctx context.Context, banProp *models.BanUserProperties) error
	OverwriteUserCache(ctx context.Context, ID string) error
	AlterDMCAStrike(ctx context.Context, ID string, delta int) error
	GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, params *models.ChannelFilterParams) ([]models.ChannelProperties, error)
	CreateUser(ctx context.Context, up *models.CreateUserProperties) (*models.Properties, error)
	GetGlobalPrivilegedUsers(ctx context.Context, roles []string) (*models.GlobalPrivilegedUsers, error)
	VerifyPhoneNumber(ctx context.Context, id, code string) error
	SetChannelProperties(ctx context.Context, up *models.UpdateChannelProperties) error
	SetBasicChannelProperties(ctx context.Context, up *models.UpdateChannelProperties) error
	HardDeleteUser(ctx context.Context, ID string, adminLogin string, skipBlock bool) error
	SoftDeleteUser(ctx context.Context, ID string, adminLogin string) error
	UndeleteUser(ctx context.Context, ID string, adminLogin string) error
	GetUserImagesByID(ctx context.Context, id string) (*models.ImageProperties, error)
	GetReservations(ctx context.Context, logins []string) (models.ReservationPropertiesResult, error)
	AddReservation(ctx context.Context, prop models.ReservationProperties) error
	UpdateReservation(ctx context.Context, prop models.ReservationProperties) error
	DeleteReservation(ctx context.Context, login string) error
}

type logicImpl struct {
	users        users.Backend
	banreport    banreport.Client
	blacklist    blacklist.Client
	partners     partnerships.Client
	rails        rails.Client
	follows      follows.Client
	kinesis      kinesis.Publisher
	spade        spade.Client
	sns          sns.Publisher
	channels     channels.Backend
	auditor      auditor.Auditor
	leviathan    leviathan.Client
	twilio       twilio.Client
	geoIP        geoip.GeoIP
	discovery    discovery.Discovery
	uploader     uploader.Uploader
	evs          evs.Client
	reservations reservations.Backend
	owl          owl.Client
	s3           s3.S3Client

	deleteAllower hardDeleteUserAllower

	seedVerifyCode string
}

var (
	loginReleaseDateNonPartner = 6 * 30 * 24 * time.Hour    // Time to block username for non partner
	loginReleaseDatePartner    = 100 * 365 * 24 * time.Hour // Indefinite
)

func (l *logicImpl) GetUserImagesByID(ctx context.Context, id string) (*models.ImageProperties, error) {
	user, err := l.users.GetUserPropertiesByID(ctx, id, ReadOptions{ReadFromMaster: false})
	if err != nil {
		return nil, err
	}

	return l.users.GetUserImages(ctx, id, *user.Login)
}

func (l *logicImpl) GetUserPropertiesByID(ctx context.Context, ID string, params *models.FilterParams) (*models.Properties, error) {
	user, err := l.users.GetUserPropertiesByID(ctx, ID, ReadOptions{ReadFromMaster: false})

	if err != nil {
		return nil, err
	}

	if !validators.ValidateProperties(*user, params) {
		return nil, models.ErrFilteredUser
	}

	return user, nil
}

func (l *logicImpl) GetUserPropertiesBulk(ctx context.Context, params *models.FilterParams) ([]models.Properties, error) {
	var filteredUsers []models.Properties
	users, err := l.users.GetUserPropertiesBulk(ctx, params)
	if err != nil {
		return nil, err
	}
	for _, user := range users {
		if !validators.ValidateProperties(user, params) {
			continue
		}

		filteredUsers = append(filteredUsers, user)
	}

	return filteredUsers, nil
}

func (l *logicImpl) GetBannedUsers(ctx context.Context, until time.Time) ([]models.Properties, error) {
	return l.users.GetBannedUsers(ctx, until)
}

func (l *logicImpl) GetUserPropertiesLike(ctx context.Context, field string, pattern string) ([]models.Properties, error) {
	return l.users.GetUserPropertiesLike(ctx, field, pattern)
}

func (l *logicImpl) isDisplaynameUnique(ctx context.Context, dn *string) bool {
	_, err := l.users.GetUserPropertiesByDisplayname(ctx, dn)
	if errx.Unwrap(err) == ErrNoProperties {
		return true
	}
	return false
}

func (l *logicImpl) AlterDMCAStrike(ctx context.Context, ID string, delta int) error {
	cup, err := l.users.GetUserPropertiesByID(ctx, ID, ReadOptions{ReadFromMaster: false})
	if err != nil {
		return ErrNoProperties
	}

	err = l.users.AlterDMCAStrike(ctx, ID, delta)
	if err != nil {
		return err
	}

	if err := l.rails.DeleteCache(ctx, cup.ID, nil); err != nil {
		return err
	}

	return nil
}

func (l *logicImpl) OverwriteUserCache(ctx context.Context, ID string) error {
	bgCtx := backend.DetachContext(ctx)
	g, ctx := errgroup.WithContext(ctx)

	g.Go(func() error {
		_, err := l.users.GetUserPropertiesByID(bgCtx, ID, ReadOptions{ReadFromMaster: true, OverwriteCache: true})
		return err
	})

	g.Go(func() error {
		// TODO: Update channels backend to treat ids as string
		intID, err := strconv.Atoi(ID)
		if err != nil {
			logx.Error(bgCtx, "failed to expire channels cache, couldn't convert id to id", logx.Fields{
				"error": err.Error(),
				"id":    ID,
			})
		}
		_, err = l.channels.GetAllChannelPropertiesBulk(bgCtx, []uint64{uint64(intID)}, nil, ReadOptions{ReadFromMaster: true, OverwriteCache: true})
		return err
	})

	return g.Wait()
}

func (l *logicImpl) overwriteUserCacheByLogin(ctx context.Context, login string) error {
	bgCtx := backend.DetachContext(ctx)
	g, ctx := errgroup.WithContext(ctx)

	g.Go(func() error {
		_, err := l.users.GetUserPropertiesByLogin(bgCtx, login, ReadOptions{ReadFromMaster: true, OverwriteCache: true})
		return err
	})

	g.Go(func() error {
		_, err := l.channels.GetAllChannelPropertiesBulk(bgCtx, nil, []string{login}, ReadOptions{ReadFromMaster: true, OverwriteCache: true})
		return err
	})

	return g.Wait()
}

func (l *logicImpl) GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, params *models.ChannelFilterParams) ([]models.ChannelProperties, error) {
	chnls, err := l.channels.GetAllChannelPropertiesBulk(ctx, channelIDs, channelNames, ReadOptions{})
	if err != nil {
		return nil, err
	}
	var filteredChannels []models.ChannelProperties
	for _, chnl := range chnls {
		if !validators.ValidateChannelProperties(chnl, params) {
			continue
		}

		filteredChannels = append(filteredChannels, chnl)
	}

	if len(chnls) > 0 && len(filteredChannels) == 0 {
		return nil, models.ErrBannedUserChannels
	}
	return filteredChannels, nil
}

func (l *logicImpl) GetGlobalPrivilegedUsers(ctx context.Context, roles []string) (*models.GlobalPrivilegedUsers, error) {
	return l.users.GetGlobalPrivilegedUsers(ctx, roles)
}

func New(u users.Backend, brc banreport.Client, bc blacklist.Client, cc rails.Client, follows follows.Client, kc kinesis.Publisher, partners partnerships.Client, spade spade.Client, sns sns.Publisher, channels channels.Backend, auditor auditor.Auditor, twi twilio.Client, leviathan leviathan.Client, discovery discovery.Discovery, geoIP geoip.GeoIP, uploader uploader.Uploader, evs evs.Client, reservations reservations.Backend, owl owl.Client, s3 s3.S3Client, seedVerifyCode string) (Logic, error) {

	l := &logicImpl{
		users:          u,
		banreport:      brc,
		blacklist:      bc,
		rails:          cc,
		follows:        follows,
		kinesis:        kc,
		partners:       partners,
		spade:          spade,
		sns:            sns,
		channels:       channels,
		auditor:        auditor,
		twilio:         twi,
		leviathan:      leviathan,
		geoIP:          geoIP,
		discovery:      discovery,
		uploader:       uploader,
		evs:            evs,
		reservations:   reservations,
		owl:            owl,
		s3:             s3,
		seedVerifyCode: seedVerifyCode,
	}

	l.deleteAllower = l
	return l, nil
}
