package logic

import (
	"errors"
	"fmt"
	"testing"
	"time"

	"golang.org/x/net/context"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	banreport "code.justin.tv/web/users-service/internal/clients/banreport/mocks"
	blacklist "code.justin.tv/web/users-service/internal/clients/blacklist/mocks"
	follows "code.justin.tv/web/users-service/internal/clients/follows/mocks"
	rails "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	mocksns "code.justin.tv/web/users-service/internal/clients/sns/mocks"
	"code.justin.tv/web/users-service/models"

	leviathan "code.justin.tv/chat/leviathan/client"
	"code.justin.tv/web/users-service/backend/channels"
	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	"code.justin.tv/web/users-service/backend/users"
	usersMocks "code.justin.tv/web/users-service/backend/users/mocks"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/auditor/mocks"
	leviathanMocks "code.justin.tv/web/users-service/internal/clients/leviathan/mocks"

	spade "code.justin.tv/common/spade-client-go/spade"
	geoipMocks "code.justin.tv/web/users-service/internal/clients/geoip/mocks"
	spadeMocks "code.justin.tv/web/users-service/internal/clients/spade/mocks"
)

var (
	geoIP = &geoipMocks.GeoIP{}
)

func TestBanUser(t *testing.T) {
	Convey("When a user is banned", t, func() {
		ctx := context.Background()
		targetID := "1"
		users, followsMock, rails, banreport, sns, channels, auditor, leviathan, spade := initSuccessfulMocks(ctx, targetID)
		logic := &logicImpl{
			follows:   followsMock,
			users:     users,
			banreport: banreport,
			rails:     rails,
			blacklist: &blacklist.Client{},
			channels:  channels,
			auditor:   auditor,
			leviathan: leviathan,
			spade:     spade,
			sns:       sns,
			geoIP:     geoIP,
		}

		Convey("Hides all follows on ban", func() {
			logic.follows = followsMock
			prop := &models.BanUserProperties{
				FromUserID:   "2",
				TargetUserID: targetID,
				Type:         "tos",
				Reason:       "meme",
				Permanent:    false,
				SkipIPBan:    true,
			}
			err := logic.BanUser(ctx, prop)
			So(err, ShouldBeNil)

			followsMock.AssertNumberOfCalls(t, "HideAllFollows", 1)
			followsMock.AssertCalled(t, "HideAllFollows", ctx, targetID, mock.Anything)
		})

		Convey("Reports errors when hiding fails", func() {
			failingFollows := initFailingFollowsMock(ctx, targetID)
			logic.follows = failingFollows
			prop := &models.BanUserProperties{
				FromUserID:   "2",
				TargetUserID: targetID,
				Type:         "tos",
				Reason:       "meme",
				Permanent:    false,
				SkipIPBan:    true,
			}
			err := logic.BanUser(ctx, prop)
			So(err, ShouldNotBeNil)

			failingFollows.AssertNumberOfCalls(t, "HideAllFollows", 1)
			failingFollows.AssertCalled(t, "HideAllFollows", ctx, targetID, mock.Anything)
		})
	})

}

func TestUnbanUser(t *testing.T) {
	Convey("When a user is unbanned", t, func() {
		ctx := context.Background()
		targetID := "1"
		users, follows, cache, banreport, mocksns, channels, auditor, leviathan, spade := initSuccessfulMocks(ctx, targetID)
		logic := &logicImpl{
			follows:   follows,
			users:     users,
			banreport: banreport,
			rails:     cache,
			blacklist: &blacklist.Client{},
			sns:       mocksns,
			channels:  channels,
			auditor:   auditor,
			leviathan: leviathan,
			spade:     spade,
			geoIP:     geoIP,
		}

		Convey("Restores all follows", func() {
			err := logic.UnbanUser(ctx, targetID, false)
			So(err, ShouldBeNil)

			follows.AssertNumberOfCalls(t, "RestoreAllFollows", 1)
			follows.AssertCalled(t, "RestoreAllFollows", ctx, targetID, mock.Anything)
		})

		Convey("Reports error when follow restoration fails", func() {
			failingFollows := initFailingFollowsMock(ctx, targetID)
			logic.follows = failingFollows
			err := logic.UnbanUser(ctx, targetID, false)
			So(err, ShouldNotBeNil)

			failingFollows.AssertNumberOfCalls(t, "RestoreAllFollows", 1)
			failingFollows.AssertCalled(t, "RestoreAllFollows", ctx, targetID, mock.Anything)
		})
	})
}

func initFailingFollowsMock(ctx context.Context, targetID string) *follows.Client {
	followsMock := follows.Client{}
	followsMock.On("RestoreAllFollows", ctx, targetID, mock.Anything).Return(errors.New("lulz"))
	followsMock.On("HideAllFollows", ctx, targetID, mock.Anything).Return(errors.New("lulz"))
	return &followsMock
}

func initSuccessfulMocks(ctx context.Context, targetID string) (users.Backend, *follows.Client, *rails.Client, *banreport.Client, *mocksns.Publisher, channels.Backend, auditor.Auditor, leviathan.Client, spade.Client) {
	usersMock := usersMocks.Backend{}
	usersMock.On("UnbanUser", ctx, targetID, mock.Anything, mock.Anything).Return(nil)
	usersMock.On("BanUser", ctx, targetID, true, true, 0, 1).Return(nil)
	usersMock.On("GetUserPropertiesByID", ctx, targetID, mock.Anything).Return(&models.Properties{}, nil)

	followsMock := follows.Client{}
	followsMock.On("RestoreAllFollows", ctx, targetID, mock.Anything).Return(nil)
	followsMock.On("HideAllFollows", ctx, targetID, mock.Anything).Return(nil)

	railsMock := rails.Client{}
	railsMock.On("DeleteCache", mock.Anything, targetID, mock.Anything).Return(nil)
	railsMock.On("VodSync", mock.Anything, targetID, mock.Anything).Return(nil)

	banreportMock := banreport.Client{}
	banreportMock.On("ReportBan", ctx, targetID, "2", "tos", true, "meme", mock.Anything).Return(nil)

	snsMock := mocksns.Publisher{}
	snsMock.On("PublishUnban", mock.Anything, mock.Anything).Return(nil)
	snsMock.On("PublishBan", mock.Anything, mock.Anything).Return(nil)
	snsMock.On("PublishBanUser", mock.Anything, mock.Anything).Return(nil)

	channelsMock := channelsMocks.Backend{}

	auditorMock := mocks.Auditor{}
	auditorMock.On("Audit", mock.Anything, mock.Anything)

	leviathanMock := leviathanMocks.Client{}
	leviathanMock.On("TosSuspensionReport", ctx, mock.Anything).Return(nil)

	spadeMock := spadeMocks.Client{}
	spadeMock.On("TrackEvent", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	return &usersMock, &followsMock, &railsMock, &banreportMock, &snsMock, &channelsMock, &auditorMock, &leviathanMock, &spadeMock
}

func TestBulkUsers(t *testing.T) {
	Convey("when bulk getting users", t, func() {
		usersMock := &usersMocks.Backend{}

		now := time.Now()
		tru := true
		users := []models.Properties{
			{
				ID:        "123",
				DeletedOn: &now,
			},
			{
				ID:            "124",
				DmcaViolation: &tru,
			},
			{
				ID: "125",
				TermsOfServiceViolation: &tru,
			},
			{
				ID: "126",
			},
		}

		usersMock.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return(users, nil)

		l := &logicImpl{
			users: usersMock,
		}

		ctx := context.Background()

		for name, scenario := range map[string]struct {
			Params        *models.FilterParams
			ExpectedUsers int
		}{
			"happy path": {
				Params:        nil,
				ExpectedUsers: 4,
			},
			"deleted": {
				Params: &models.FilterParams{
					NotDeleted: true,
				},
				ExpectedUsers: 3,
			},
			"tos": {
				Params: &models.FilterParams{
					NoTOSViolation: true,
				},
				ExpectedUsers: 3,
			},
			"dmca": {
				Params: &models.FilterParams{
					NoDMCAViolation: true,
				},
				ExpectedUsers: 3,
			},
		} {
			Convey(fmt.Sprintf("testing %s", name), func() {
				users, err := l.GetUserPropertiesBulk(ctx, scenario.Params)
				So(err, ShouldBeNil)

				So(len(users), ShouldEqual, scenario.ExpectedUsers)
			})
		}
	})
}

func TestUsersByID(t *testing.T) {
	Convey("when getting a users by id", t, func() {
		usersMock := &usersMocks.Backend{}

		now := time.Now()
		tru := true

		l := &logicImpl{
			users: usersMock,
		}

		ctx := context.Background()

		for name, scenario := range map[string]struct {
			User    *models.Properties
			Params  *models.FilterParams
			Failure bool
		}{
			"happy path": {
				User: &models.Properties{
					ID: "123",
				},
				Params:  nil,
				Failure: false,
			},
			"deleted": {
				User: &models.Properties{
					ID:        "124",
					DeletedOn: &now,
				},
				Params: &models.FilterParams{
					NotDeleted: true,
				},
				Failure: true,
			},
			"tos": {
				User: &models.Properties{
					ID: "125",
					TermsOfServiceViolation: &tru,
				},
				Params: &models.FilterParams{
					NoTOSViolation: true,
				},
				Failure: true,
			},
			"dmca": {
				User: &models.Properties{
					ID:            "126",
					DmcaViolation: &tru,
				},
				Params: &models.FilterParams{
					NoDMCAViolation: true,
				},
				Failure: true,
			},
		} {
			Convey(fmt.Sprintf("testing %s", name), func() {

				usersMock.On("GetUserPropertiesByID", mock.Anything, mock.Anything, mock.Anything).Return(scenario.User, nil)

				_, err := l.GetUserPropertiesByID(ctx, scenario.User.ID, scenario.Params)
				if scenario.Failure {
					So(err, ShouldNotBeNil)
				} else {
					So(err, ShouldBeNil)
				}
			})
		}
	})
}
