package logic

import (
	"time"

	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"

	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
)

const (
	login_block_record   = "Block Record"
	login_user_record    = "User"
	login_invalid_record = "Invalid Record"
	jtvReservation       = "JTV Reservation"
	jtvPartnerRelease    = "2017-11-09T11:00:00-08:00"
	jtvAffiliateRelease  = "2017-11-14T11:00:00-08:00"
	jtvGeneralRelease    = "2017-11-17T11:00:00-08:00"

	RenameUserBlockRecordType         = "User Block Record"
	RenameUserReservedRecordReason    = "User Rename"
	RenamePartnerBlockRecordType      = "Partner Block Record"
	RenamePartnerReservedRecordReason = "Partner Rename"

	// For blocked path which we'll never want to override even with admin access
	ReservedPathBlockType = "Reserved Record"
)

func (l *logicImpl) getLoginPropertiesForLogins(ctx context.Context, logins []string, overrideLoginLength bool) ([]models.LoginTypeProperties, error) {
	lps, err := l.users.GetLogins(ctx, logins)
	if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
		return nil, err
	}

	lpr := []models.LoginTypeProperties{}
	validLogins := []string{}
	for _, login := range logins {
		if err := validators.IsLoginValid(&login, overrideLoginLength); err != nil {
			lpr = append(lpr, models.LoginTypeProperties{Login: login, Type: login_invalid_record})
		} else {
			validLogins = append(validLogins, login)
		}
	}

	for _, lp := range lps {
		var loginType string
		if lp.RenamedLoginBlockRecord != nil && *lp.RenamedLoginBlockRecord {
			loginType = login_block_record
		} else {
			loginType = login_user_record
		}
		lpr = append(lpr, models.LoginTypeProperties{ID: lp.ID, Login: *lp.Login, Type: loginType})
	}

	return lpr, nil
}

func (l *logicImpl) getReservationPropertiesForLogins(ctx context.Context, logins []string) ([]models.ReservationProperties, error) {
	rps, err := l.reservations.GetReservations(ctx, logins)
	if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
		return nil, err
	}
	return rps, nil
}

func (l *logicImpl) appendUnexpiredReservations(ctx context.Context, lpr []models.LoginTypeProperties, rps []models.ReservationProperties) ([]models.LoginTypeProperties, bool) {
	hasChanged := false
	currentTime := time.Now()
	for _, lp := range rps {
		// A login being returned means that the login should be considered unavailable for registration or renaming
		// Names are returned if they haven't expired or never expire, though we ignore jtv reservations
		if lp.Type != jtvReservation {
			if lp.ExpiresOn == nil || (*lp.ExpiresOn).After(currentTime) {
				lpr = append(lpr, models.LoginTypeProperties{Login: lp.Login, Type: lp.Type})
			} else {
				hasChanged = true
			}
		}
	}

	return lpr, hasChanged
}

func (l *logicImpl) appendJtvReservations(ctx context.Context, lpr []models.LoginTypeProperties, rps []models.ReservationProperties, isRename bool, isPartnered bool, isAffiliated bool) ([]models.LoginTypeProperties, bool, error) {
	hasChanged := false
	currentTime := time.Now()
	partnerJtvReleaseTime, err := time.Parse(time.RFC3339, jtvPartnerRelease)
	if err != nil {
		return nil, hasChanged, err
	}
	affiliateJtvReleaseTime, err := time.Parse(time.RFC3339, jtvAffiliateRelease)
	if err != nil {
		return nil, hasChanged, err
	}
	generalJtvReleaseTime, err := time.Parse(time.RFC3339, jtvGeneralRelease)
	if err != nil {
		return nil, hasChanged, err
	}

	releaseTime := generalJtvReleaseTime
	if isPartnered {
		releaseTime = partnerJtvReleaseTime
	} else if isAffiliated {
		releaseTime = affiliateJtvReleaseTime
	}

	jtvReleaseReady := isRename && currentTime.After(releaseTime)
	for _, lp := range rps {
		// Stopgap to not release JTV Reservations early
		// If the JTV release is not ready for the user, do not make the login available
		// Catch non-JTV reservations and make sure they are not bypassed here
		if lp.Type == jtvReservation {
			if !jtvReleaseReady {
				lpr = append(lpr, models.LoginTypeProperties{Login: lp.Login, Type: lp.Type})
			} else {
				hasChanged = true
			}
		}
	}

	return lpr, hasChanged, nil
}

func (l *logicImpl) GetTypeForLogins(ctx context.Context, logins []string, requestingUserID string, isRename bool, overrideLoginLength bool) ([]models.LoginTypeProperties, bool, error) {
	var g errgroup.Group
	var err error
	var lpr []models.LoginTypeProperties
	var rps []models.ReservationProperties
	var hadJtvReservations bool
	isPartnered := false
	isAffiliated := false

	if requestingUserID != "" {
		g.Go(func() error {
			var partnerErr error
			payoutType, partnerErr := l.partners.GetUserPayoutType(ctx, requestingUserID, nil)
			isPartnered = payoutType.IsPartner
			isAffiliated = payoutType.IsAffiliate
			return partnerErr
		})
	}

	g.Go(func() error {
		var loginErr error
		lpr, loginErr = l.getLoginPropertiesForLogins(ctx, logins, overrideLoginLength)
		return loginErr
	})

	g.Go(func() error {
		var reservationErr error
		rps, reservationErr = l.getReservationPropertiesForLogins(ctx, logins)
		return reservationErr
	})

	xErr := g.Wait()
	if xErr != nil {
		return nil, false, xErr
	}

	lpr, _ = l.appendUnexpiredReservations(ctx, lpr, rps)
	lpr, hadJtvReservations, err = l.appendJtvReservations(ctx, lpr, rps, isRename, isPartnered, isAffiliated)
	if err != nil {
		return nil, false, err
	}

	return lpr, hadJtvReservations, nil
}
