package logic

import (
	"fmt"
	"math/rand"
	"time"

	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/models"

	"code.justin.tv/chat/golibs/logx"
	"golang.org/x/net/context"
)

func verifyCode(seedVerifyCode string) string {
	if !configs.IsProduction() && seedVerifyCode != "" {
		return seedVerifyCode
	}

	rand.Seed(time.Now().Unix())
	code := rand.Intn(1000000)
	return fmt.Sprintf("%06.f", float64(code))
}

func (l *logicImpl) sendSMS(ctx context.Context, number, body string) error {
	if !configs.IsProduction() && l.twilio == nil {
		return nil
	}

	return l.twilio.SendSMS(ctx, number, body)
}

func verifyPhoneNumberProperties(props *models.PhoneNumberProperties, code string, received time.Time) error {
	if props.SentAt.Add(1 * time.Hour).Local().Before(received.Local()) {
		return models.ErrVerifyCodeExpired
	}

	if props.Code != code {
		return models.ErrVerifyDoesNotMatch
	}

	return nil
}

func (l *logicImpl) sendVerificationCode(ctx context.Context, id, number, code string) error {
	// Allow users to zero out number
	if number == "" {
		return nil
	}

	msg := fmt.Sprintf("Hello from Twitch! Your verification code is %s. It expires in 1 hour.", code)
	return l.sendSMS(ctx, number, msg)
}

func (l *logicImpl) VerifyPhoneNumber(ctx context.Context, id, code string) error {
	props, err := l.users.GetUserPhoneNumber(ctx, id)
	if err != nil {
		return err
	}

	if err := verifyPhoneNumberProperties(props, code, time.Now()); err != nil {
		return err
	}

	if err := l.users.VerifyUserPhoneNumber(ctx, id); err != nil {
		return err
	}

	if err := l.rails.DeleteCache(ctx, id, nil); err != nil {
		logx.Error(ctx, err, logx.Fields{"id": id})
	}

	bCtx := backend.DetachContext(ctx)
	go func() {
		if err := l.sendSMS(bCtx, props.PhoneNumber, "Thanks for verifying your phone number on Twitch. You can now receive SMS alerts."); err != nil {
			logx.Error(bCtx, fmt.Sprintf("failed to send sms: %s", err), logx.Fields{
				"error": err.Error(),
			})
		}
		if err := l.sendSMS(bCtx, props.PhoneNumber, "You are responsible for any text message service fees. Visit https://twitch.tv/settings to stop these messages."); err != nil {
			logx.Error(bCtx, fmt.Sprintf("failed to send sms: %s", err), logx.Fields{
				"error": err.Error(),
			})
		}
	}()

	return nil
}
