package logic

import (
	"time"

	"golang.org/x/net/context"

	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/backend/util"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
)

func (l *logicImpl) GetRenameEligibility(ctx context.Context, ID string) (*models.RenameProperties, error) {
	up, err := l.users.GetUserPropertiesByID(ctx, ID, ReadOptions{ReadFromMaster: false})
	if err != nil {
		return nil, err
	}
	payoutType, err := l.partners.GetUserPayoutType(ctx, ID, nil)
	if err != nil {
		return nil, err
	}
	isPartnered := payoutType.IsPartner
	isAffiliated := payoutType.IsAffiliate

	return l.IsRenameEligible(ctx, up.LastLoginChangeDate, isPartnered, isAffiliated)
}

func (l *logicImpl) IsRenameEligible(ctx context.Context, lastLoginChangeDate *time.Time, isPartnered bool, isAffiliated bool) (*models.RenameProperties, error) {
	var isEligible bool
	var eligibleOn time.Time

	if lastLoginChangeDate != nil {
		var releaseTime time.Time
		var err error
		if isPartnered {
			releaseTime, err = time.Parse(time.RFC3339, jtvPartnerRelease)
			if err != nil {
				return nil, err
			}
		} else if isAffiliated {
			releaseTime, err = time.Parse(time.RFC3339, jtvAffiliateRelease)
			if err != nil {
				return nil, err
			}
		} else {
			releaseTime, err = time.Parse(time.RFC3339, jtvGeneralRelease)
			if err != nil {
				return nil, err
			}
		}
		lastRename := lastLoginChangeDate
		eligibleOn = lastRename.Add(time.Hour * 24 * models.LoginRenameCooldown)

		//Special logic to grant free renames when time is at or after the release time
		if !time.Now().Before(releaseTime) {
			//If the user has not changed their login since release, then they get the free rename
			if lastLoginChangeDate.Before(releaseTime) {
				eligibleOn = releaseTime
			}
		}

		isEligible = !time.Now().Before(eligibleOn)
	} else {
		eligibleOn = time.Now()
		isEligible = true
	}

	rr := &models.RenameProperties{
		RenameEligible:   &isEligible,
		RenameEligibleOn: &eligibleOn,
	}

	return rr, nil
}

func (l *logicImpl) rollbackReservedRecord(ctx context.Context, oldLogin string, previousReservedRecord *models.ReservationProperties, currentReservedRecord *models.ReservationProperties) error {
	reservation, err := l.reservations.GetReservation(ctx, oldLogin)
	if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
		return err
	}

	// If the record has already been deleted or rolled back, skip
	if reservation == nil || currentReservedRecord == nil {
		return models.ErrTryToRollbackDifferentBlockRecord
	}

	// Make sure the record in db has same expire time with the one we want to delete to avoid bad race condition
	if reservationEqual(*reservation, *currentReservedRecord) {
		// If the new record is overriding an existing one, rollback to the existing one
		if previousReservedRecord != nil && previousReservedRecord.Login != "" {
			err = l.reservations.UpdateReservation(ctx, *previousReservedRecord)
		} else {
			err = l.reservations.DeleteReservation(ctx, oldLogin)
		}
		if err != nil {
			return err
		}
	}

	return models.ErrTryToRollbackDifferentBlockRecord
}

func reservationEqual(previousReservedRecord models.ReservationProperties, currentReservedRecord models.ReservationProperties) bool {
	if previousReservedRecord.ExpiresOn == nil && currentReservedRecord.ExpiresOn == nil {
		return true
	}

	if previousReservedRecord.ExpiresOn != nil && currentReservedRecord.ExpiresOn != nil {
		roundA := (*previousReservedRecord.ExpiresOn).Round(time.Second)
		roundB := (*currentReservedRecord.ExpiresOn).Round(time.Second)

		if !roundA.Equal(roundB) {
			return false
		}

		if previousReservedRecord.Type != currentReservedRecord.Type {
			return false
		}

		if previousReservedRecord.Reason != currentReservedRecord.Reason {
			return false
		}

		if previousReservedRecord.Login != currentReservedRecord.Login {
			return false
		}

		return true
	}

	return false
}

func reportRename(ctx context.Context) {
	stats := config.ClusterStatsd()
	metric := "user_rename.all"
	if err := stats.Inc(metric, 1, 1.0); err != nil {
		logx.Warn(ctx, fmt.Sprintf("failed to report stats counter for %q", metric))
	}
}

func reportJtvRename(ctx context.Context) {
	stats := config.ClusterStatsd()
	metric := "user_rename.jtv"
	if err := stats.Inc(metric, 1, 1.0); err != nil {
		logx.Warn(ctx, fmt.Sprintf("failed to report stats counter for %q", metric))
	}
}

func (l *logicImpl) validateUpdatableLogin(ctx context.Context, cup *models.Properties, upUpdate *models.UpdateableProperties, reservedRecordProperty *models.ReservationProperties) (bool, error) {
	err := validators.IsLoginValid(upUpdate.NewLogin, upUpdate.OverrideLoginLength)
	if err != nil {
		return false, err
	}

	payoutType, err := l.partners.GetUserPayoutType(ctx, cup.ID, nil)
	if err != nil {
		return false, err
	}
	isPartnered := payoutType.IsPartner
	isAffiliated := payoutType.IsAffiliate

	if isPartnered {
		upUpdate.ReleaseDateDuration = loginReleaseDatePartner
		reservedRecordProperty.Reason = RenamePartnerReservedRecordReason
		reservedRecordProperty.Type = RenamePartnerBlockRecordType
	} else {
		upUpdate.ReleaseDateDuration = loginReleaseDateNonPartner
		reservedRecordProperty.Reason = RenameUserReservedRecordReason
		reservedRecordProperty.Type = RenameUserBlockRecordType
		releaseDate := time.Now().Add(upUpdate.ReleaseDateDuration).UTC()
		reservedRecordProperty.ExpiresOn = &releaseDate
	}

	reservedRecordProperty.Login = strings.ToLower(*cup.Login)
	rep, err := l.IsRenameEligible(ctx, cup.LastLoginChangeDate, isPartnered, isAffiliated)
	if err != nil {
		return false, err
	}

	if upUpdate.SkipLoginCooldown == nil || (upUpdate.SkipLoginCooldown != nil && !*upUpdate.SkipLoginCooldown) {
		if rep.RenameEligible == nil || !*rep.RenameEligible {
			return false, models.ErrNotAllowedToChangeLogin
		}
	}

	// Login should always be lowercase.
	*upUpdate.NewLogin = strings.ToLower(*upUpdate.NewLogin)

	result, err := l.users.GetUserBlock(ctx, *upUpdate.NewLogin)
	if err != nil {
		return false, err
	}

	if result != nil {
		if result.RenamedLoginBlockRecord == nil || *result.RenamedLoginBlockRecord == false {
			return false, models.ErrLoginNotAvailable
		}

		if upUpdate.OverrideLoginBlock == nil || *upUpdate.OverrideLoginBlock == false {
			return false, models.ErrLoginBlocked
		}
	}
	// We need to know here that we're being allowed to claim a JTV login
	rps, hadJtvReservation, err := l.GetTypeForLogins(ctx, []string{*upUpdate.NewLogin}, cup.ID, true, upUpdate.OverrideLoginLength)
	if err != nil {
		return false, err
	}

	if len(rps) != 0 {
		// Never want to override reserved path record even with admin access
		for _, rp := range rps {
			if rp.Type == ReservedPathBlockType {
				return false, models.ErrLoginBlocked
			}
		}

		if upUpdate.OverrideLoginBlock == nil || *upUpdate.OverrideLoginBlock == false {
			return false, models.ErrLoginBlocked
		}
	}

	if !validators.IsDisplaynameCJK(cup.Displayname) {
		upUpdate.Displayname = upUpdate.NewLogin
	}

	return hadJtvReservation, nil
}
