package logic

import (
	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	reservationMocks "code.justin.tv/web/users-service/backend/reservations/mocks"
	usersMocks "code.justin.tv/web/users-service/backend/users/mocks"
	auditorMocks "code.justin.tv/web/users-service/internal/clients/auditor/mocks"
	kinesisMocks "code.justin.tv/web/users-service/internal/clients/kinesis/mocks"
	partnerMocks "code.justin.tv/web/users-service/internal/clients/partnerships/mocks"
	rails "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	mocksns "code.justin.tv/web/users-service/internal/clients/sns/mocks"
	spadeMocks "code.justin.tv/web/users-service/internal/clients/spade/mocks"

	"context"
	"testing"

	"time"

	"code.justin.tv/revenue/moneypenny/client"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestRenames(t *testing.T) {
	newLogin := "newLogin"
	pastTime := time.Now().Truncate(time.Hour)
	for _, scenario := range []struct {
		Name                    string
		PreviousUserProperties  models.Properties
		UpdatableUserProperties *models.UpdateableProperties
		UserBlocks              *models.BlockProperties
		UserLogins              []models.LoginProperties
		Reservations            []models.ReservationProperties
		Reservation             *models.ReservationProperties
		IsPartner               bool
		Error                   error
	}{
		{
			Name:                   "happy path with normal user renmane",
			Error:                  nil,
			IsPartner:              false,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{},
			Reservation:            &models.ReservationProperties{},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin: &newLogin,
			},
		},
		{
			Name:                   "happy path with normal user renmane",
			Error:                  nil,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{},
			Reservation:            &models.ReservationProperties{},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin: &newLogin,
			},
		},
		{
			Name:                   "normal user rename with reservation db already has the login",
			Error:                  models.ErrLoginBlocked,
			IsPartner:              false,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: nil, Type: RenameUserBlockRecordType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: nil, Type: RenameUserBlockRecordType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin: &newLogin,
			},
		},
		{
			Name:                   "partenr rename with reservation db already has the login",
			Error:                  models.ErrLoginBlocked,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: nil, Type: RenamePartnerBlockRecordType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: nil, Type: RenamePartnerBlockRecordType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin: &newLogin,
			},
		},
		{
			Name:                   "normal user rename with reservation db already has the login but has override tag in request",
			Error:                  nil,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: nil, Type: RenameUserBlockRecordType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: nil, Type: RenameUserBlockRecordType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
		{
			Name:                   "partner rename with reservation db already has the login but has override tag in request",
			Error:                  nil,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: nil, Type: RenamePartnerBlockRecordType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: nil, Type: RenamePartnerBlockRecordType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
		{
			Name:                   "rename with reservation db already has the login as reserved path but has override tag in request",
			Error:                  models.ErrLoginBlocked,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: nil, Type: ReservedPathBlockType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: nil, Type: ReservedPathBlockType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
		{
			Name:                   "rename with reservation db already has the login but expired",
			Error:                  nil,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{{Login: "newLogin", ExpiresOn: &pastTime, Type: RenamePartnerBlockRecordType, Reason: "testing"}},
			Reservation:            &models.ReservationProperties{Login: "newLogin", ExpiresOn: &pastTime, Type: RenamePartnerBlockRecordType, Reason: "testing"},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             nil,
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
		{
			Name:                   "rename with site db has the user record will fail the request even with override",
			Error:                  models.ErrLoginNotAvailable,
			IsPartner:              true,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{},
			Reservation:            &models.ReservationProperties{},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             &models.BlockProperties{Login: &newLogin, RenamedLoginBlockRecord: aws.Bool(false), ID: "123"},
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
		{
			Name:                   "rename with site db has the block record without override",
			Error:                  models.ErrLoginBlocked,
			IsPartner:              false,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{},
			Reservation:            &models.ReservationProperties{},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             &models.BlockProperties{Login: &newLogin, RenamedLoginBlockRecord: aws.Bool(true), ID: "123"},
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin: &newLogin,
			},
		},
		{
			Name:                   "rename with site db has the block record with override",
			Error:                  nil,
			IsPartner:              false,
			PreviousUserProperties: *Users[AdminUserID],
			Reservations:           []models.ReservationProperties{},
			Reservation:            &models.ReservationProperties{},
			UserLogins:             []models.LoginProperties{},
			UserBlocks:             &models.BlockProperties{Login: &newLogin, RenamedLoginBlockRecord: aws.Bool(true), ID: "123"},
			UpdatableUserProperties: &models.UpdateableProperties{
				NewLogin:           &newLogin,
				OverrideLoginBlock: aws.Bool(true),
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			ctx := context.Background()
			userID := scenario.PreviousUserProperties.ID

			railsMock := rails.Client{}
			railsMock.On("DeleteCache", mock.Anything, userID, mock.Anything).Return(nil)

			snsMock := mocksns.Publisher{}
			snsMock.On("PublishChannelUpdate", mock.Anything, mock.Anything).Return(nil)
			snsMock.On("PublishRename", mock.Anything, mock.Anything).Return(nil)
			snsMock.On("PublishUpdate", mock.Anything, mock.Anything).Return(nil)

			usersMock := usersMocks.Backend{}
			usersMock.On("GetUserPropertiesByID", mock.Anything, userID, mock.Anything).Return(Users[userID], nil)
			usersMock.On("GetUserBlock", ctx, "newlogin").Return(scenario.UserBlocks, nil)
			usersMock.On("GetLogins", ctx, []string{"newlogin"}).Return(scenario.UserLogins, nil)
			usersMock.On("UpdateProperties", ctx, mock.Anything, mock.Anything).Return(nil)

			auditorMock := auditorMocks.Auditor{}
			auditorMock.On("Audit", mock.Anything, mock.Anything)

			spadeMock := spadeMocks.Client{}
			spadeMock.On("TrackEvent", mock.Anything, mock.Anything, mock.Anything).Return(nil)

			channelsMock := channelsMocks.Backend{}
			channelsMock.On("GetChannelsByRedirectChannel", ctx, *Users[userID].Login).Return([]models.ChannelProperties{}, nil)
			channelsMock.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{}, nil)
			channelsMock.On("ExpireChannelProperties", mock.Anything, mock.Anything).Return(nil)

			partnerMocks := partnerMocks.Client{}
			partnerResponse := &moneypenny.GetUserPayoutTypeResponse{IsAffiliate: false}
			if scenario.IsPartner {
				partnerResponse.IsPartner = true
			} else {
				partnerResponse.IsPartner = false
			}
			partnerMocks.On("GetUserPayoutType", mock.Anything, userID, mock.Anything).Return(partnerResponse, nil)

			reservationMocks := reservationMocks.Backend{}
			reservationMocks.On("GetReservations", ctx, mock.Anything).Return(scenario.Reservations, nil)
			reservationMocks.On("GetReservation", ctx, *Users[AdminUserID].Login).Return(scenario.Reservation, nil)
			reservationMocks.On("UpdateReservation", ctx, mock.Anything).Return(nil)
			reservationMocks.On("DeleteReservation", ctx, "newlogin").Return(nil)

			kinesisMocks := kinesisMocks.Publisher{}
			kinesisMocks.On("Publish", mock.Anything, mock.Anything)

			logic := &logicImpl{
				rails:        &railsMock,
				sns:          &snsMock,
				users:        &usersMock,
				auditor:      &auditorMock,
				spade:        &spadeMock,
				channels:     &channelsMock,
				kinesis:      &kinesisMocks,
				reservations: &reservationMocks,
				partners:     &partnerMocks,
			}

			err := logic.SetUserProperties(ctx, scenario.PreviousUserProperties.ID, scenario.UpdatableUserProperties)
			assert.Equal(t, err, scenario.Error)
		})
	}
}
