package logic

import (
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (l *logicImpl) GetReservations(ctx context.Context, logins []string) (models.ReservationPropertiesResult, error) {
	lowercasedLogins := []string{}
	for _, login := range logins {
		lowercasedLogins = append(lowercasedLogins, strings.ToLower(login))
	}
	reservations, err := l.reservations.GetReservations(ctx, lowercasedLogins)
	return models.ReservationPropertiesResult{reservations}, errx.New(err)
}

func (l *logicImpl) AddReservation(ctx context.Context, prop models.ReservationProperties) error {
	prop.Login = strings.ToLower(prop.Login)
	reservation, err := l.reservations.GetReservation(ctx, prop.Login)
	if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
		return err
	}

	if reservation != nil {
		return models.ErrReservationAlreadyExist
	}

	return l.reservations.AddReservation(ctx, prop)
}

func (l *logicImpl) DeleteReservation(ctx context.Context, login string) error {
	login = strings.ToLower(login)
	reservation, err := l.reservations.GetReservation(ctx, login)
	if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
		return err
	}
	if reservation == nil || err == util.ErrNoProperties {
		return models.ErrReservationNotExist
	}
	return l.reservations.DeleteReservation(ctx, login)
}

func (l *logicImpl) UpdateReservation(ctx context.Context, prop models.ReservationProperties) error {
	prop.Login = strings.ToLower(prop.Login)
	reservation, err := l.reservations.GetReservation(ctx, prop.Login)
	if err != nil && err != util.ErrNoProperties {
		return err
	}

	if reservation == nil || errx.Unwrap(err) == util.ErrNoProperties {
		return models.ErrReservationNotExist
	}

	return l.reservations.UpdateReservation(ctx, prop)
}
