package logic

import (
	"testing"

	"golang.org/x/net/context"

	"errors"
	"time"

	reservationMocks "code.justin.tv/web/users-service/backend/reservations/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/mock"
)

func TestAddReservation(t *testing.T) {
	prop := models.ReservationProperties{}
	backendErr := errors.New("SQL error")
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, l *logicImpl)
		Prop  models.ReservationProperties
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(nil, nil)
				reservationMock.On("AddReservation", mock.Anything, prop).Return(nil)
				l.reservations = reservationMock
			},
			Prop:  prop,
			Error: nil,
		},
		{
			Name: "reservation already exist during add new reservation",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&prop, nil)
				l.reservations = reservationMock
			},
			Prop:  prop,
			Error: models.ErrReservationAlreadyExist,
		},
		{
			Name: "backend returns error",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(nil, nil)
				reservationMock.On("AddReservation", mock.Anything, prop).Return(backendErr)
				l.reservations = reservationMock
			},
			Prop:  prop,
			Error: backendErr,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.AddReservation(context.Background(), scenario.Prop)
			if err != scenario.Error {
				t.Fatalf("errors do not match: %v vs %v", err, scenario.Error)
			}
		})
	}
}

func TestDeleteReservation(t *testing.T) {
	prop := models.ReservationProperties{}
	backendErr := errors.New("SQL error")
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, l *logicImpl)
		prop  models.ReservationProperties
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&prop, nil)
				reservationMock.On("DeleteReservation", mock.Anything, prop.Login).Return(nil)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: nil,
		},
		{
			Name: "reservation doesn't exist during add new reservation",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(nil, nil)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: models.ErrReservationNotExist,
		},
		{
			Name: "backend returns error",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&prop, nil)
				reservationMock.On("DeleteReservation", mock.Anything, prop.Login).Return(backendErr)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: backendErr,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.DeleteReservation(context.Background(), scenario.prop.Login)
			if err != scenario.Error {
				t.Fatalf("errors do not match: %v vs %v", err, scenario.Error)
			}
		})
	}
}

func TestUpdateReservation(t *testing.T) {
	now := time.Now()
	addMinute := time.Now().Add(time.Minute)
	prop := models.ReservationProperties{ExpiresOn: &addMinute}
	backendErr := errors.New("SQL error")
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, l *logicImpl)
		prop  models.ReservationProperties
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&models.ReservationProperties{}, nil)
				reservationMock.On("UpdateReservation", mock.Anything, prop).Return(nil)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: nil,
		},
		{
			Name: "reservation exists but with different expiration time",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&models.ReservationProperties{ExpiresOn: &now}, nil)
				reservationMock.On("UpdateReservation", mock.Anything, prop).Return(nil)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: nil,
		},
		{
			Name: "reservation doesn't exist during add new reservation",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(nil, nil)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: models.ErrReservationNotExist,
		},
		{
			Name: "backend returns error",
			Setup: func(t *testing.T, l *logicImpl) {
				reservationMock := &reservationMocks.Backend{}
				reservationMock.On("GetReservation", mock.Anything, prop.Login).Return(&models.ReservationProperties{}, nil)
				reservationMock.On("UpdateReservation", mock.Anything, prop).Return(backendErr)
				l.reservations = reservationMock
			},
			prop:  prop,
			Error: backendErr,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			l := &logicImpl{}

			if scenario.Setup != nil {
				scenario.Setup(t, l)
			}

			err := l.UpdateReservation(context.Background(), scenario.prop)
			if err != scenario.Error {
				t.Fatalf("errors do not match: %v vs %v", err, scenario.Error)
			}
		})
	}
}
