package logic

import (
	"strconv"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
	"golang.org/x/net/context"
)

func (l *logicImpl) SetChannelProperties(ctx context.Context, up *models.UpdateChannelProperties) error {
	cup, err := l.channels.GetAllChannelPropertiesBulk(ctx, []uint64{up.ID}, []string{}, ReadOptions{})
	if err != nil || len(cup) == 0 {
		return ErrNoProperties
	}

	err = l.validateUpdateableChannelProperties(ctx, &cup[0], up)
	if err != nil {
		return err
	}

	uup := up.FillFromProperties(&cup[0])

	err = l.channels.UpdateChannelProperties(ctx, *uup)
	if err != nil {
		return err
	}

	l.publishChannelUpdates(ctx, &cup[0], up)

	return nil
}

func (l *logicImpl) SetBasicChannelProperties(ctx context.Context, up *models.UpdateChannelProperties) error {
	cup, err := l.channels.GetAllChannelPropertiesBulk(ctx, []uint64{up.ID}, []string{}, ReadOptions{})
	if err != nil || len(cup) == 0 {
		return ErrNoProperties
	}

	err = l.validateUpdateableChannelProperties(ctx, &cup[0], up)
	if err != nil {
		return err
	}

	uup := up.FillFromProperties(&cup[0])

	err = l.channels.UpdateBasicChannelProperties(ctx, *uup)
	if err != nil {
		return err
	}

	l.publishChannelUpdates(ctx, &cup[0], up)
	return nil
}

func (l *logicImpl) validateUpdateableChannelProperties(ctx context.Context, cup *models.ChannelProperties, up *models.UpdateChannelProperties) error {
	err := validators.IsBroadcasterLanguageValid(up.BroadcasterLanguage)
	if err != nil {
		return err
	}

	err = validators.IsStatusValid(up.Status)
	if err != nil {
		return err
	}

	if up.Game != nil {
		err = validators.IsGameValid(up.Game)
		if err != nil {
			return err
		}

		if up.GameID == nil {
			gid, err := l.discovery.GetGameIDByName(ctx, *up.Game)
			if gid != 0 && err == nil {
				gameID := uint64(gid)
				up.GameID = &gameID
			}
		}
	}

	return nil
}

func (l *logicImpl) publishChannelUpdates(ctx context.Context, original *models.ChannelProperties, changes *models.UpdateChannelProperties) {
	bgCtx := backend.DetachContext(ctx)

	if original == nil || changes == nil {
		return
	}

	sid := strconv.FormatUint(changes.ID, 10)
	changes.ID = original.ID

	go func() {
		event := models.SNSChannelUpdateEvent{
			Original:  original,
			Changes:   changes,
			Timestamp: time.Now(),
		}
		if err := l.sns.PublishChannelUpdate(bgCtx, event); err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	go func() {
		err := l.spade.TrackEvent(bgCtx, "channel_update", models.UpdateChannelEvent{
			ChannelID: sid,
			Channel:   original.Name,
			OldStatus: original.Status,
			Status:    changes.Status,
			OldGame:   original.Game,
			Game:      changes.Game,
			OldBroadcasterLanguage: original.BroadcasterLanguage,
			BroadcasterLanguage:    changes.BroadcasterLanguage,
			OldGameID:              original.GameID,
			GameID:                 changes.GameID,
			Env:                    configs.Environment(),
		})
		if err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	if err := l.rails.DeleteCache(ctx, strconv.FormatUint(changes.ID, 10), nil); err != nil {
		logx.Error(ctx, err, logx.Fields{"id": changes.ID})
	}
}
