package logic

import (
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"

	rails "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	mocksns "code.justin.tv/web/users-service/internal/clients/sns/mocks"

	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/auditor/mocks"

	"testing"

	spade "code.justin.tv/common/spade-client-go/spade"
	spadeMocks "code.justin.tv/web/users-service/internal/clients/spade/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

type SetChannelsTest struct {
	suite.Suite
	logic    *logicImpl
	channels *channelsMocks.Backend
	rails    *rails.Client
	sns      *mocksns.Publisher
	auditor  auditor.Auditor
	spade    spade.Client
	ctx      context.Context
}

func (suite *SetChannelsTest) SetupTest() {
	suite.ctx = context.Background()
	suite.rails, suite.sns, suite.channels, suite.auditor, suite.spade = initSuccessfulMocksForSetChannels(suite.ctx, AdminUserID)
	suite.logic = &logicImpl{
		rails:    suite.rails,
		channels: suite.channels,
		auditor:  suite.auditor,
		spade:    suite.spade,
		sns:      suite.sns,
	}
}

func (suite *SetChannelsTest) TestSetChannelSuccess() {
	up := models.UpdateChannelProperties{
		ID:     Channels[AdminUserID].ID,
		Status: Channels[ToSetPropUserID].Status,
	}
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{Channels[AdminUserID]}, nil)
	channel := Channels[AdminUserID]
	uup := up.FillFromProperties(&channel)
	suite.channels.On("UpdateChannelProperties", mock.Anything, *uup).Return(nil)
	err := suite.logic.SetChannelProperties(suite.ctx, &up)
	assert.NoError(suite.T(), err)
	suite.channels.AssertExpectations(suite.T())
}

func initSuccessfulMocksForSetChannels(ctx context.Context, targetID string) (*rails.Client, *mocksns.Publisher, *channelsMocks.Backend, auditor.Auditor, spade.Client) {
	railsMock := rails.Client{}
	railsMock.On("DeleteCache", ctx, targetID, mock.Anything).Return(nil)
	snsMock := mocksns.Publisher{}
	snsMock.On("PublishChannelUpdate", mock.Anything, mock.Anything).Return(nil)
	channelsMock := channelsMocks.Backend{}
	auditorMock := mocks.Auditor{}
	auditorMock.On("Audit", mock.Anything, mock.Anything)
	spade := spadeMocks.Client{}
	spade.On("TrackEvent", mock.Anything, "channel_update", mock.Anything).Return(nil)
	return &railsMock, &snsMock, &channelsMock, &auditorMock, &spade
}

func TestSetChannelSuite(t *testing.T) {
	suite.Run(t, new(SetChannelsTest))
}
