package logic

import (
	"time"

	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (l *logicImpl) SetUserImageProperties(ctx context.Context, updatableImageProp *models.ImageProperties) error {
	user, err := l.users.GetUserPropertiesByID(ctx, updatableImageProp.ID, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return util.ErrNoProperties
	}

	images, err := l.users.GetUserImages(ctx, updatableImageProp.ID, *user.Login)
	if err != nil {
		return err
	}

	updates := fillImageProps(updatableImageProp, images)
	if err != nil {
		return errx.New(err)
	}

	err = l.users.SetUserImageProperties(ctx, updates)
	if err != nil {
		return err
	}

	err = l.OverwriteUserCache(ctx, updatableImageProp.ID)
	if err != nil {
		return err
	}

	err = l.rails.DeleteCache(ctx, updatableImageProp.ID, nil)
	if err != nil {
		return err
	}

	l.publishUserImageUpdates(ctx, user, images, updatableImageProp)
	return nil
}

func (l *logicImpl) publishUserImageUpdates(ctx context.Context, user *models.Properties, originalImages, updatableImageProp *models.ImageProperties) {
	bgCtx := backend.DetachContext(ctx)
	changedImageType := "remove_image"
	previousImage := "customized_image"
	if updatableImageProp.DefaultProfileImage != nil && *updatableImageProp.DefaultProfileImage != image.ClearProfileImage {
		changedImageType = image.DefaultProfileImage
	} else if updatableImageProp.DefaultProfileBanner != nil && *updatableImageProp.DefaultProfileBanner != image.ClearProfileBanner {
		changedImageType = image.DefaultProfileBanner
	} else if updatableImageProp.DefaultChannelOfflineImage != nil && *updatableImageProp.DefaultChannelOfflineImage != image.ClearChannelOfflineImage {
		changedImageType = image.DefaultChannelOfflineImage
	} else if updatableImageProp.ProfileImage != nil {
		changedImageType = models.ProfilePictureImageType
	} else if updatableImageProp.ProfileBanner != nil {
		changedImageType = models.ProfileBannerImageType
	} else if updatableImageProp.ChannelOfflineImage != nil {
		changedImageType = models.ChannelOfflineImageType
	}

	switch changedImageType {
	case models.ProfilePictureImageType, image.DefaultProfileImage:
		if originalImages.DefaultProfileImage != nil {
			previousImage = *originalImages.DefaultProfileImage
		}
	case models.ProfileBannerImageType, image.DefaultProfileBanner:
		if originalImages.DefaultProfileBanner != nil {
			previousImage = *originalImages.DefaultProfileBanner
		}
	case models.ChannelOfflineImageType, image.DefaultChannelOfflineImage:
		if originalImages.DefaultChannelOfflineImage != nil {
			previousImage = *originalImages.DefaultChannelOfflineImage
		}
	}

	images, err := l.users.GetUserImages(ctx, user.ID, *user.Login)
	if err != nil {
		logx.Error(ctx, err)
		return
	}

	if updatableImageProp.ProfileImage != nil {
		updatableImageProp.ProfileImage = images.ProfileImage
	}

	if updatableImageProp.ProfileBanner != nil {
		updatableImageProp.ProfileBanner = images.ProfileBanner
	}

	if updatableImageProp.ChannelOfflineImage != nil {
		updatableImageProp.ChannelOfflineImage = images.ChannelOfflineImage
	}

	go func() {
		event := events.UpdateImageEvent(updatableImageProp.ID, *updatableImageProp)
		l.auditor.Audit(bgCtx, event)
	}()

	go func() {
		err := l.spade.TrackEvent(bgCtx, "set_user_images", models.SetUserImagesEvent{
			UserID: updatableImageProp.ID,
			DefaultChannelOfflineImageID: updatableImageProp.DefaultChannelOfflineImage,
			DefaultProfileImageID:        updatableImageProp.DefaultProfileImage,
			DefaultProfileBannerID:       updatableImageProp.DefaultProfileBanner,
			ImageType:                    changedImageType,
			PreviousImage:                previousImage,
			Env:                          configs.Environment(),
		})
		if err != nil {
			logx.Error(bgCtx, "failed to report spade set user images event", logx.Fields{
				"error": err,
			})
		}
	}()

	go func() {
		event := models.SNSUpdateImageEvent{
			UserID:    updatableImageProp.ID,
			Original:  user,
			Changed:   updatableImageProp,
			Timestamp: time.Now(),
		}
		if err := l.sns.PublishImageUpdate(bgCtx, event); err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	go func() {
		l.deleteImages(bgCtx, user.ID, *user.Login, originalImages, updatableImageProp)
	}()

	go func() {
		if metricErr := config.ClusterStatsd().Inc(fmt.Sprintf("%s.update", changedImageType), 1, 1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics %s: %s", changedImageType, metricErr))
		}
	}()
}

func fillImageProps(i, prop *models.ImageProperties) models.ImageProperties {
	updates := models.ImageProperties{
		ID:                         i.ID,
		ProfileImage:               i.ProfileImage,
		ChannelOfflineImage:        i.ChannelOfflineImage,
		ProfileBanner:              i.ProfileBanner,
		DefaultProfileBanner:       i.DefaultProfileBanner,
		DefaultChannelOfflineImage: i.DefaultChannelOfflineImage,
		DefaultProfileImage:        i.DefaultProfileImage,
	}

	if prop == nil {
		return updates
	}

	if updates.DefaultProfileImage == nil && updates.ProfileImage == nil {
		updates.ProfileImage = prop.ProfileImage
	}

	if updates.DefaultChannelOfflineImage == nil && updates.ChannelOfflineImage == nil {
		updates.ChannelOfflineImage = prop.ChannelOfflineImage
	}

	if updates.DefaultProfileBanner == nil && updates.ProfileBanner == nil {
		updates.ProfileBanner = prop.ProfileBanner
	}

	return updates
}

func (l *logicImpl) UploadUserImages(ctx context.Context, updatableImageProp *models.UploadableImage) (models.UploadInfo, error) {
	user, err := l.users.GetUserPropertiesByID(ctx, updatableImageProp.ID, util.ReadOptions{ReadFromMaster: false})
	if err != nil || user == nil {
		return models.UploadInfo{}, util.ErrNoProperties
	}

	return l.uploader.UploadImages(ctx, *updatableImageProp)
}

func (l *logicImpl) deleteImages(ctx context.Context, id, login string, original, update *models.ImageProperties) {
	paths := []string{}

	if update.DefaultProfileImage != nil {
		if original.DefaultProfileImage == nil && original.ProfileImage != nil {
			for _, image := range *original.ProfileImage {
				path, e := yimg.ProfileImagesName(image, login)
				if e == nil {
					paths = append(paths, path)
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.profile_image.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}
	}

	if update.DefaultProfileBanner != nil {
		if original.DefaultProfileBanner == nil && original.ProfileBanner != nil {
			for _, image := range *original.ProfileBanner {
				path, e := yimg.ProfileBannersName(image, login)
				if e == nil {
					paths = append(paths, path)
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.profile_banner.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}
	}

	if update.DefaultChannelOfflineImage != nil {
		if original.DefaultChannelOfflineImage == nil && original.ChannelOfflineImage != nil {
			for _, image := range *original.ChannelOfflineImage {
				path, e := yimg.ChannelImageName(image, login)
				if e == nil {
					paths = append(paths, path)
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.channel_offline_image.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}
	}

	if len(paths) != 0 {
		err := l.s3.BatchDelete(paths)
		if err != nil && !utils.IsHystrixErr(err) {
			logx.Error(ctx, err, logx.Fields{
				"paths": paths,
			})
		}
	}
}
