package logic

import (
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"

	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	usersMocks "code.justin.tv/web/users-service/backend/users/mocks"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/auditor/mocks"
	rails "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	s3Mocks "code.justin.tv/web/users-service/internal/clients/s3/mocks"
	mocksns "code.justin.tv/web/users-service/internal/clients/sns/mocks"

	"testing"

	"time"

	spade "code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/users-service/backend/util"
	spadeMocks "code.justin.tv/web/users-service/internal/clients/spade/mocks"
	uploaderMocks "code.justin.tv/web/users-service/internal/clients/uploader/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

type SetUserImageTest struct {
	suite.Suite
	logic    *logicImpl
	users    *usersMocks.Backend
	rails    *rails.Client
	channels *channelsMocks.Backend
	sns      *mocksns.Publisher
	auditor  auditor.Auditor
	spade    spade.Client
	ctx      context.Context
	uploader *uploaderMocks.Uploader
	s3       *s3Mocks.S3Client
}

func (suite *SetUserImageTest) SetupTest() {
	suite.ctx = context.Background()
	suite.rails, suite.sns, suite.users, suite.auditor, suite.spade, suite.channels, suite.uploader, suite.s3 = initSuccessfulMocksForSetUserImage(suite.ctx, AdminUserID)
	suite.logic = &logicImpl{
		rails:    suite.rails,
		users:    suite.users,
		auditor:  suite.auditor,
		spade:    suite.spade,
		sns:      suite.sns,
		channels: suite.channels,
		uploader: suite.uploader,
		s3:       suite.s3,
	}
}

func (suite *SetUserImageTest) TestSetUserImageSuccess() {
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false}).Return(Users[AdminUserID], nil)
	input := &models.ImageProperties{
		ID:           AdminUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	suite.users.On("GetUserImages", mock.Anything, AdminUserID, *Users[AdminUserID].Login).Return(input, nil)
	suite.users.On("SetUserImageProperties", mock.Anything, *input).Return(nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(nil, nil)
	suite.rails.On("DeleteCache", mock.Anything, AdminUserID, nil).Return(nil)
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{}, nil)
	err := suite.logic.SetUserImageProperties(suite.ctx, input)
	assert.NoError(suite.T(), err)
}

func (suite *SetUserImageTest) TestUploadImageSuccess() {
	uploadableImage := models.UploadableImage{
		ID:   AdminUserID,
		Type: "profile_image",
	}
	suite.uploader.On("UploadImages", mock.Anything, uploadableImage).Return(models.UploadInfo{"validid", "validurl"}, nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false, OverwriteCache: false}).Return(Users[AdminUserID], nil)
	s, err := suite.logic.UploadUserImages(suite.ctx, &uploadableImage)
	assert.NoError(suite.T(), err)
	assert.NotEmpty(suite.T(), s)
}

func (suite *SetUserImageTest) TestUploadImageWithUserNotFound() {
	uploadableImage := models.UploadableImage{
		ID:   AdminUserID,
		Type: "profile_image",
	}
	suite.uploader.On("UploadImages", mock.Anything, uploadableImage).Return(models.UploadInfo{"validid", "validurl"}, nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false, OverwriteCache: false}).Return(nil, nil)
	_, err := suite.logic.UploadUserImages(suite.ctx, &uploadableImage)
	assert.NotNil(suite.T(), err)
}

func (suite *SetUserImageTest) TestUserDefaultImageSuccess() {
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false}).Return(Users[AdminUserID], nil)
	input := &models.ImageProperties{
		ID:           AdminUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	defaultImage := "profile_image_001"
	update := &models.ImageProperties{
		ID:                  AdminUserID,
		DefaultProfileImage: &defaultImage,
	}

	suite.users.On("GetUserImages", mock.Anything, AdminUserID, *Users[AdminUserID].Login).Return(input, nil)
	suite.users.On("SetUserImageProperties", mock.Anything, mock.Anything).Return(nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(nil, nil)
	suite.rails.On("DeleteCache", mock.Anything, AdminUserID, mock.Anything).Return(nil)
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{}, nil)
	suite.s3.On("BatchDelete", mock.Anything).Return(nil)
	err := suite.logic.SetUserImageProperties(suite.ctx, update)

	time.Sleep(2 * time.Second)
	assert.NoError(suite.T(), err)
	suite.users.AssertExpectations(suite.T())
	suite.s3.AssertExpectations(suite.T())
	suite.rails.AssertExpectations(suite.T())
	suite.channels.AssertExpectations(suite.T())
}

func (suite *SetUserImageTest) TestUserDefaultImageSuccessWithoutPreviousImage() {
	emptyImageUser := CopyUser(Users[AdminUserID])
	emptyImageUser.ProfileImage = nil

	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false}).Return(emptyImageUser, nil)
	input := &models.ImageProperties{
		ID: AdminUserID,
	}

	defaultImage := "profile_image_001"
	update := &models.ImageProperties{
		ID:                  AdminUserID,
		DefaultProfileImage: &defaultImage,
	}

	suite.users.On("GetUserImages", mock.Anything, AdminUserID, *Users[AdminUserID].Login).Return(update, nil)
	suite.users.On("SetUserImageProperties", mock.Anything, *input).Return(nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(nil, nil)
	suite.rails.On("DeleteCache", mock.Anything, AdminUserID, mock.Anything).Return(nil)
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{}, nil)
	err := suite.logic.SetUserImageProperties(suite.ctx, input)

	time.Sleep(2 * time.Second)
	assert.NoError(suite.T(), err)
	suite.users.AssertExpectations(suite.T())
	suite.s3.AssertExpectations(suite.T())
	suite.rails.AssertExpectations(suite.T())
	suite.channels.AssertExpectations(suite.T())
}

func (suite *SetUserImageTest) TestUserDefaultImageSuccessWithPreviousAlsoDefaultImage() {
	defaultImageUser := CopyUser(Users[AdminUserID])
	defaultImage := "profile_image_001"

	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: false}).Return(defaultImageUser, nil)
	input := &models.ImageProperties{
		ID:                  AdminUserID,
		ProfileImage:        Users[AdminUserID].ProfileImage,
		DefaultProfileImage: &defaultImage,
	}

	update := &models.ImageProperties{
		ID:                  AdminUserID,
		DefaultProfileImage: &defaultImage,
	}

	suite.users.On("GetUserImages", mock.Anything, AdminUserID, *Users[AdminUserID].Login).Return(update, nil)
	suite.users.On("SetUserImageProperties", mock.Anything, *input).Return(nil)
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(nil, nil)
	suite.rails.On("DeleteCache", mock.Anything, AdminUserID, mock.Anything).Return(nil)
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{}, nil)
	err := suite.logic.SetUserImageProperties(suite.ctx, input)

	time.Sleep(2 * time.Second)
	assert.NoError(suite.T(), err)
	suite.users.AssertExpectations(suite.T())
	suite.s3.AssertExpectations(suite.T())
	suite.rails.AssertExpectations(suite.T())
	suite.channels.AssertExpectations(suite.T())
}

func initSuccessfulMocksForSetUserImage(ctx context.Context, targetID string) (*rails.Client, *mocksns.Publisher, *usersMocks.Backend, auditor.Auditor, spade.Client, *channelsMocks.Backend, *uploaderMocks.Uploader, *s3Mocks.S3Client) {
	railsMock := rails.Client{}
	railsMock.On("DeleteCache", ctx, targetID, mock.Anything).Return(nil)
	snsMock := mocksns.Publisher{}
	snsMock.On("PublishImageUpdate", mock.Anything, mock.Anything).Return(nil)
	usersMock := usersMocks.Backend{}
	auditorMock := mocks.Auditor{}
	auditorMock.On("Audit", mock.Anything, mock.Anything)
	spade := spadeMocks.Client{}
	spade.On("TrackEvent", mock.Anything, "set_user_images", mock.Anything).Return(nil)
	channels := channelsMocks.Backend{}
	uploaderMocks := uploaderMocks.Uploader{}
	s3Mocks := s3Mocks.S3Client{}
	return &railsMock, &snsMock, &usersMock, &auditorMock, &spade, &channels, &uploaderMocks, &s3Mocks
}

func TestSetUserImageSuite(t *testing.T) {
	suite.Run(t, new(SetUserImageTest))
}
