package logic

import (
	"fmt"
	"net/http"
	"reflect"
	"strings"
	"time"

	"code.justin.tv/foundation/twitchclient"

	"strconv"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	evsmodels "code.justin.tv/growth/emailvalidator/models"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/internal/clients/kinesis"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
	"golang.org/x/net/context"
)

func (l *logicImpl) SetUserProperties(ctx context.Context, ID string, upUpdate *models.UpdateableProperties) error {
	cup, err := l.users.GetUserPropertiesByID(ctx, ID, ReadOptions{ReadFromMaster: true})
	if err != nil {
		return ErrNoProperties
	}

	hadJtvReservation := false
	reservedRecordProperty := models.ReservationProperties{}
	if upUpdate.NewLogin != nil {
		hadJtvReservation, err = l.validateUpdatableLogin(ctx, cup, upUpdate, &reservedRecordProperty)
		if err != nil {
			return err
		}
	}

	if err := l.validateUpdateableProperties(ctx, cup, upUpdate); err != nil {
		return err
	}
	sendEmail := l.shouldSendVerificationEmail(ctx, cup, upUpdate)

	// Upsert a reserved record to reservation db when this is a approved rename
	previousReservedRecord := &models.ReservationProperties{}
	if upUpdate.NewLogin != nil && *upUpdate.NewLogin != *cup.Login {
		previousReservedRecord, err = l.reservations.GetReservation(ctx, *cup.Login)
		if err != nil && errx.Unwrap(err) != util.ErrNoProperties {
			return err
		}

		if previousReservedRecord == nil || previousReservedRecord.Type != ReservedPathBlockType {
			err = l.reservations.UpdateReservation(ctx, reservedRecordProperty)
			if err != nil {
				return err
			}
		}
	}

	err = l.users.UpdateProperties(ctx, upUpdate, cup)
	if err != nil {
		// Rollback reservation db change during a failure to rename user
		if upUpdate.NewLogin != nil && *upUpdate.NewLogin != *cup.Login {
			reservationDBErr := l.rollbackReservedRecord(ctx, *cup.Login, previousReservedRecord, &reservedRecordProperty)
			if reservationDBErr != nil {
				logx.Error(ctx, reservationDBErr)
			}
		}
		return err
	}

	bgCtx := backend.DetachContext(ctx)
	if upUpdate.NewLogin != nil && *upUpdate.NewLogin != *cup.Login {
		err = l.reservations.DeleteReservation(ctx, *upUpdate.NewLogin)
		if err != nil {
			logx.Error(ctx, err)
		}
		// Redirect channel update and publish
		channels, err := l.channels.GetChannelsByRedirectChannel(ctx, *cup.Login)
		if err != nil {
			logx.Error(ctx, "action: update the redirect for channel pointing to a renamed user", logx.Fields{
				"error":        err.Error(),
				"renamed_user": *cup.Login,
				"new_login":    *upUpdate.NewLogin,
			})
		} else {
			for _, channel := range channels {
				err = l.SetChannelProperties(ctx, &models.UpdateChannelProperties{
					ID:              channel.ID,
					RedirectChannel: upUpdate.NewLogin,
				})
				if err != nil {
					logx.Error(ctx, "action: update the redirect for channel pointing to a renamed user", logx.Fields{
						"error":           err.Error(),
						"renamed_user":    *cup.Login,
						"new_login":       *upUpdate.NewLogin,
						"from_channel_id": channel.ID,
					})
				}
			}
		}

		go func() {
			u, err := strconv.ParseUint(cup.ID, 10, 64)
			if err != nil {
				logx.Error(bgCtx, err)
				return
			}

			event := models.SNSChannelUpdateEvent{
				Original:  &models.ChannelProperties{ID: u, Name: *cup.Login},
				Changes:   &models.UpdateChannelProperties{ID: u, Login: upUpdate.NewLogin},
				Timestamp: time.Now(),
			}
			if err := l.sns.PublishChannelUpdate(bgCtx, event); err != nil {
				logx.Error(bgCtx, err)
			}
		}()
	}

	//at this point we know things worked, and can surface stats
	if sendEmail {
		go func() {
			login := upUpdate.NewLogin
			if login == nil {
				login = cup.Login
			}

			language := upUpdate.Language
			if language == nil {
				language = cup.Language
			}

			// Create the updated user object for sending verification email
			uup := models.Properties{
				ID:       ID,
				Login:    login,
				Email:    upUpdate.Email,
				Language: language,
			}

			// send verification email
			err = l.sendNewUserVerificationEmailToEVS(bgCtx, uup)
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}()
	}

	return l.publishUpdate(ctx, cup, upUpdate, hadJtvReservation)
}

//will need to return source of login
//break login validation into its own function?
func (l *logicImpl) validateUpdateableProperties(ctx context.Context, cup *models.Properties, upUpdate *models.UpdateableProperties) error {
	err := validators.IsEmailValid(upUpdate.Email, cup)
	if err != nil {
		return err
	}

	// Disable this check until we can fully understand the performance impact
	// if upUpdate.Email != nil && (cup.Email == nil || !strings.EqualFold(*upUpdate.Email, *cup.Email)) {
	// 	//For non whitelisted accounts, ensure that they can't reuse capped email
	// 	if !validators.IsWhitelistedEmail(upUpdate.Email) {
	// 		users, err := l.GetUserPropertiesBulk(ctx, &models.FilterParams{Emails: []string{*upUpdate.Email}})
	// 		if err != nil {
	// 			return err
	// 		}
	// 		if len(users) > maxAllowed {
	// 			return models.TooManyUsersAssociatedWithEmail
	// 		}
	// 	}

	// }

	err = validators.IsLanguageValid(upUpdate.Language)
	if err != nil {
		return err
	}

	err = validators.IsDescriptionValid(upUpdate.Description)
	if err != nil {
		return err
	}

	err = validators.IsPhoneNumberValid(upUpdate.PhoneNumber)
	if err != nil {
		return err
	}

	// Only validate displayname if new login is not equal to new displayname
	if !(upUpdate.NewLogin == upUpdate.Displayname) {
		if upUpdate.OverrideDisplayname {
			err = validators.IsDisplaynameValidForOverride(cup, upUpdate.Displayname)
			if err != nil {
				return err
			}
		} else {
			err = validators.IsDisplaynameValid(upUpdate.Displayname, cup)
			if err != nil {
				return err
			}
		}

		if upUpdate.Displayname != nil && !strings.EqualFold(*upUpdate.Displayname, *cup.Displayname) {
			if !l.isDisplaynameUnique(ctx, upUpdate.Displayname) {
				return models.ErrDisplaynameNotAvailable
			}
		}
	}

	if upUpdate.LastLogin != nil {
		if err := validators.IsLastLoginValid(*upUpdate.LastLogin); err != nil {
			return err
		}
	}

	if upUpdate.PhoneNumber != nil {
		upUpdate.PhoneNumberCode = verifyCode(l.seedVerifyCode)
	}

	if upUpdate.RemoteIP != nil && *upUpdate.RemoteIP != "" {
		location, _ := l.geoIP.GetCountry(*upUpdate.RemoteIP)
		upUpdate.Location = &location
	}

	return nil
}

func (l *logicImpl) publishUpdate(ctx context.Context, cup *models.Properties, upUpdate *models.UpdateableProperties, jtvReservation bool) error {
	bgCtx := backend.DetachContext(ctx)

	if cup == nil || upUpdate == nil {
		return nil
	}

	upUpdate.ID = cup.ID

	// Expire new entry
	ID := cup.ID
	err := l.OverwriteUserCache(ctx, ID)
	if err != nil {
		logx.Error(ctx, err)
	}

	// Expire old record
	if upUpdate.NewLogin != nil && *upUpdate.NewLogin != *cup.Login {
		channelToExpire := &models.ChannelProperties{Name: *cup.Login}
		if u, err := strconv.ParseUint(cup.ID, 10, 64); err == nil {
			channelToExpire.ID = u
		}
		if err = l.channels.ExpireChannelProperties(ctx, channelToExpire); err != nil {
			logx.Error(ctx, err)
		}
	}

	if err := l.rails.DeleteCache(ctx, cup.ID, nil); err != nil {
		logx.Error(ctx, err)
	}

	if upUpdate.NewLogin != nil {

		reportRename(ctx)
		renameType := "normal"
		if jtvReservation {
			renameType = "jtv"
			reportJtvRename(ctx)
		}

		go func() {
			l.kinesis.Publish(bgCtx, kinesis.NewLoginChangedEvent(ID, cup, upUpdate))
		}()
		go func() {
			err := l.spade.TrackEvent(bgCtx, "login_rename", models.LoginRenameEvent{
				UserID:   cup.ID,
				OldLogin: *cup.Login,
				NewLogin: *upUpdate.NewLogin,
				Type:     renameType,
				Env:      configs.Environment(),
			})
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}()
		go func() {
			err := l.sns.PublishRename(bgCtx, models.NewRenameEvent(ID, cup, upUpdate))
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}()
	}
	if upUpdate.Displayname != nil && *upUpdate.Displayname != *cup.Displayname {
		go func() {
			l.kinesis.Publish(bgCtx, kinesis.NewDisplaynameChangedEvent(ID, cup, upUpdate))
		}()
		go func() {
			t := "displayname_change"
			if upUpdate.NewLogin != nil {
				t = "login_rename"
			}
			err := l.spade.TrackEvent(ctx, "displayname_change_intl", models.DisplaynameChangeEvent{
				UserID:             cup.ID,
				OldDisplayname:     utils.StringValue(cup.Displayname),
				OldDisplaynameLang: validators.GetDisplaynameLang(cup.Displayname),
				NewDisplayname:     utils.StringValue(upUpdate.Displayname),
				NewDisplaynameLang: validators.GetDisplaynameLang(upUpdate.Displayname),
				UserLanguage:       utils.StringValue(cup.Language),
				Type:               t,
				Env:                configs.Environment(),
			})
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}()
		go func() {
			err := l.sns.PublishRename(bgCtx, models.NewRenameEvent(ID, cup, upUpdate))
			if err != nil {
				logx.Error(bgCtx, err)
			}
		}()
	}

	go func() {
		updateEvents := events.UpdateUserEvent(cup, upUpdate)
		l.auditor.Audit(bgCtx, updateEvents)
	}()

	if upUpdate.PhoneNumber != nil {
		go func() {
			if err := l.sendVerificationCode(bgCtx, cup.ID, *upUpdate.PhoneNumber, upUpdate.PhoneNumberCode); err != nil && err != models.ErrInvalidPhoneNumber {
				logx.Error(bgCtx, fmt.Sprintf("failed to send verify code text message: %s", err))
			}
		}()
	}

	go func() {
		e := models.SNSUpdateEvent{
			UserID:    cup.ID,
			Original:  cup,
			Changed:   upUpdate,
			Timestamp: time.Now(),
		}
		if err := l.sns.PublishUpdate(ctx, e); err != nil {
			logx.Error(ctx, err)
		}
	}()

	return nil
}

// sendVerificationEmail decides if we need to send a verification email to the user
// The verification email is sent if any of the following occurs:
// 	- The email was updated
//	- The only updated attributes are Email and EmailVerified
// For both cases above the new email should not previously have a verifed/rejected
// status for the user in EVS
func (l *logicImpl) shouldSendVerificationEmail(ctx context.Context, cup *models.Properties, upUpdate *models.UpdateableProperties) bool {
	if upUpdate.Email == nil || (cup.Email != nil && strings.EqualFold(*upUpdate.Email, *cup.Email) && cup.EmailVerified != nil && *cup.EmailVerified) {
		return false
	}

	verificationOnlyUpdate := models.UpdateableProperties{
		Email:         upUpdate.Email,
		EmailVerified: upUpdate.EmailVerified,
	}

	if reflect.DeepEqual(*upUpdate, verificationOnlyUpdate) || cup.Email == nil || !strings.EqualFold(*upUpdate.Email, *cup.Email) {
		ev := false
		// Fetch the verification status from EVS
		verificationReq, err := l.evs.VerificationStatus(ctx, EmailValidationNamespace, cup.ID, *upUpdate.Email, nil)
		if err != nil {
			verifyErr, ok := err.(*twitchclient.Error)
			if !ok || verifyErr.StatusCode != http.StatusNotFound {
				// Error - Log and treat this as unverified
				logx.Error(ctx, "failed to validate user email status", logx.Fields{
					"error": err.Error(),
				})
			}
			upUpdate.EmailVerified = &ev
			return true
		}
		if verificationReq.Status == evsmodels.StatusVerifiedString {
			// email already verified for this ID. set status to true and don't send email
			ev = true
			upUpdate.EmailVerified = &ev
			return false
		} else if verificationReq.Status == evsmodels.StatusRejectedString {
			// email was rejected, set status to false and don't send email
			upUpdate.EmailVerified = &ev
			return false
		} else {
			// Status is pending, set email verified to false and resend email
			upUpdate.EmailVerified = &ev
			return true
		}
	}
	return false
}
