package logic

import (
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"

	channelsMocks "code.justin.tv/web/users-service/backend/channels/mocks"
	reservationMocks "code.justin.tv/web/users-service/backend/reservations/mocks"
	usersMocks "code.justin.tv/web/users-service/backend/users/mocks"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/auditor/mocks"
	kinesisMocks "code.justin.tv/web/users-service/internal/clients/kinesis/mocks"
	partnerMocks "code.justin.tv/web/users-service/internal/clients/partnerships/mocks"
	rails "code.justin.tv/web/users-service/internal/clients/rails/mocks"
	s3Mocks "code.justin.tv/web/users-service/internal/clients/s3/mocks"
	mocksns "code.justin.tv/web/users-service/internal/clients/sns/mocks"

	"testing"

	spade "code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/revenue/moneypenny/client"
	"code.justin.tv/web/users-service/backend/util"
	spadeMocks "code.justin.tv/web/users-service/internal/clients/spade/mocks"
	uploaderMocks "code.justin.tv/web/users-service/internal/clients/uploader/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

type SetUserTest struct {
	suite.Suite
	logic        *logicImpl
	users        *usersMocks.Backend
	rails        *rails.Client
	channels     *channelsMocks.Backend
	sns          *mocksns.Publisher
	auditor      auditor.Auditor
	spade        spade.Client
	ctx          context.Context
	uploader     *uploaderMocks.Uploader
	s3           *s3Mocks.S3Client
	partner      *partnerMocks.Client
	reservations *reservationMocks.Backend
	kinesis      *kinesisMocks.Publisher
}

func (suite *SetUserTest) SetupTest() {
	suite.ctx = context.Background()
	suite.rails, suite.sns, suite.users, suite.auditor, suite.spade, suite.channels, suite.uploader, suite.s3, suite.partner, suite.reservations, suite.kinesis = initSuccessfulMocksForSetUser(suite.ctx, AdminUserID)
	suite.logic = &logicImpl{
		rails:        suite.rails,
		users:        suite.users,
		auditor:      suite.auditor,
		spade:        suite.spade,
		sns:          suite.sns,
		channels:     suite.channels,
		uploader:     suite.uploader,
		partners:     suite.partner,
		s3:           suite.s3,
		reservations: suite.reservations,
		kinesis:      suite.kinesis,
	}
}

func (suite *SetUserTest) TestRenameUserWithRedirectChannelUpdateSuccess() {
	newLogin := "newLogin"
	suite.users.On("GetUserPropertiesByID", mock.Anything, AdminUserID, mock.Anything).Return(Users[AdminUserID], nil)
	suite.users.On("GetUserPropertiesByLogin", mock.Anything, *Users[AdminUserID].Login, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(nil, nil)
	suite.rails.On("DeleteCache", mock.Anything, AdminUserID, nil).Return(nil)
	suite.channels.On("GetChannelsByRedirectChannel", mock.Anything, mock.Anything).Return([]models.ChannelProperties{Channels[ToSetPropUserID]}, nil)
	suite.channels.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]models.ChannelProperties{Channels[AdminUserID]}, nil)
	suite.channels.On("UpdateChannelProperties", mock.Anything, mock.Anything).Return(nil)
	suite.partner.On("GetUserPayoutType", mock.Anything, AdminUserID, mock.Anything).Return(&moneypenny.GetUserPayoutTypeResponse{IsPartner: false, IsAffiliate: false}, nil)
	suite.users.On("GetUserBlock", mock.Anything, mock.Anything).Return(nil, nil)
	suite.users.On("GetLogins", mock.Anything, mock.Anything).Return([]models.LoginProperties{}, nil)
	suite.reservations.On("GetReservations", mock.Anything, mock.Anything).Return([]models.ReservationProperties{}, nil)
	suite.users.On("UpdateProperties", mock.Anything, mock.Anything, Users[AdminUserID]).Return(nil)

	err := suite.logic.SetUserProperties(suite.ctx, AdminUserID, &models.UpdateableProperties{ID: AdminUserID, NewLogin: &newLogin})
	assert.NoError(suite.T(), err)
}

func initSuccessfulMocksForSetUser(ctx context.Context, targetID string) (*rails.Client, *mocksns.Publisher, *usersMocks.Backend, auditor.Auditor, spade.Client, *channelsMocks.Backend, *uploaderMocks.Uploader, *s3Mocks.S3Client, *partnerMocks.Client, *reservationMocks.Backend, *kinesisMocks.Publisher) {
	railsMock := rails.Client{}
	railsMock.On("DeleteCache", ctx, targetID, mock.Anything).Return(nil)
	snsMock := mocksns.Publisher{}
	snsMock.On("PublishChannelUpdate", mock.Anything, mock.Anything).Return(nil)
	snsMock.On("PublishRename", mock.Anything, mock.Anything).Return(nil)
	snsMock.On("PublishUpdate", mock.Anything, mock.Anything).Return(nil)
	usersMock := usersMocks.Backend{}
	auditorMock := mocks.Auditor{}
	auditorMock.On("Audit", mock.Anything, mock.Anything)
	spade := spadeMocks.Client{}
	spade.On("TrackEvent", mock.Anything, "channel_update", mock.Anything).Return(nil)
	spade.On("TrackEvent", mock.Anything, "login_rename", mock.Anything).Return(nil)
	spade.On("TrackEvent", mock.Anything, "displayname_change_intl", mock.Anything).Return(nil)
	channels := channelsMocks.Backend{}
	channels.On("ExpireChannelProperties", mock.Anything, mock.Anything).Return(nil)
	uploaderMocks := uploaderMocks.Uploader{}
	s3Mocks := s3Mocks.S3Client{}
	partnerMocks := partnerMocks.Client{}
	reservationMocks := reservationMocks.Backend{}
	reservationMocks.On("GetReservations", ctx, mock.Anything).Return([]models.ReservationProperties{}, nil)
	reservationMocks.On("GetReservation", ctx, *Users[AdminUserID].Login).Return(nil, nil)
	reservationMocks.On("UpdateReservation", ctx, mock.Anything).Return(nil)
	reservationMocks.On("DeleteReservation", ctx, mock.Anything).Return(nil)
	kinesisMocks := kinesisMocks.Publisher{}
	kinesisMocks.On("Publish", mock.Anything, mock.Anything)
	return &railsMock, &snsMock, &usersMock, &auditorMock, &spade, &channels, &uploaderMocks, &s3Mocks, &partnerMocks, &reservationMocks, &kinesisMocks
}

func TestUserSuite(t *testing.T) {
	suite.Run(t, new(SetUserTest))
}
