package logic

import (
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/internal/clients/hystrix"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (l *logicImpl) UnbanUser(ctx context.Context, ID string, ifResetCount bool) error {
	bgCtx := backend.DetachContext(ctx)
	timestamp := time.Now()

	up, err := l.users.GetUserPropertiesByID(ctx, ID, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}

	tosCount := utils.IntValue(up.TosViolationCount)
	dmcaCount := utils.IntValue(up.DmcaViolationCount)
	if ifResetCount {
		tosCount = utils.MinValue(tosCount, 2)
		dmcaCount = utils.MinValue(dmcaCount, 2)
	}

	if err := l.users.UnbanUser(ctx, ID, tosCount, dmcaCount); err != nil {
		return err
	}

	if up.RemoteIP != nil {
		if err = l.blacklist.DeleteIP(ctx, *up.RemoteIP, nil); err != nil {
			return err
		}
	}

	go func() {
		e := models.SNSUnbanEvent{
			UserID:      ID,
			Timestamp:   timestamp,
			UpdatedUser: *up,
		}
		if err := l.sns.PublishUnban(bgCtx, e); err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	go func() {
		unbanEvent := events.UnbanUserEvent(ID)
		l.auditor.Audit(bgCtx, unbanEvent)
	}()

	if err := l.rails.DeleteCache(ctx, ID, nil); err != nil {
		return err
	}

	return hystrix.Do(hystrix.CommandRestoreAllFollows, func() error {
		return l.follows.RestoreAllFollows(ctx, ID, nil)
	}, nil)
}
