# Canary ASG to allow experimental deploys with minimized impact.

resource "template_file" "canary_app_user_data" {
  template = "${file("${path.module}/asg_puppetizer.template")}"
  vars {
    hostname_prefix = "${var.hostname}-aws"
    hostname_suffix = "${lookup(var.domain, var.environment)}"
    cluster = "${var.name}-${var.environment}"
    environment = "${var.environment}"
    canary = "true"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_launch_configuration" "canary_app_conf" {
  name_prefix = "${var.name}-canary-app"
  image_id = "${var.ami}"
  instance_type = "${var.instance_type}"
  security_groups = ["${var.security_group}"]
  user_data = "${template_file.canary_app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "canary_app_asg" {
  name = "${var.name}_${var.environment}_canary_asg"
  vpc_zone_identifier = ["${data.terraform_remote_state.twitch_target_aws_account.service_a}",
                          "${data.terraform_remote_state.twitch_target_aws_account.service_b}",
                          "${data.terraform_remote_state.twitch_target_aws_account.service_c}"]
  max_size = "${var.canary_size}"
  min_size = "${var.canary_size}"
  desired_capacity = "${var.canary_size}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.canary_app_conf.name}"

  tag {
    key = "Name"
    value = "${var.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.project}"
    propagate_at_launch = true
  }
}

module "canary_asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.name}_${var.environment}_canary_topic"
  asg_name = "${aws_autoscaling_group.canary_app_asg.name}"
  lambda_arn = "${data.terraform_remote_state.production_asg_lambda.asg_lambda_arn}"
}
