# Primary application ASG and ELB

resource "template_file" "app_user_data" {
  template = "${file("${path.module}/asg_puppetizer.template")}"
  vars {
    hostname_prefix = "${var.hostname}-aws"
    hostname_suffix = "${lookup(var.domain, var.environment)}"
    cluster = "${var.name}-${var.environment}"
    environment = "${var.environment}"
    canary = "false"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_launch_configuration" "app_conf" {
  name_prefix = "${var.name}-app"
  image_id = "${var.ami}"
  instance_type = "${var.instance_type}"
  security_groups = ["${var.security_group}"]
  user_data = "${template_file.app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "app_asg" {
  name = "${var.name}_${var.environment}_asg"
  vpc_zone_identifier = ["${data.terraform_remote_state.twitch_target_aws_account.service_a}",
                          "${data.terraform_remote_state.twitch_target_aws_account.service_b}",
                          "${data.terraform_remote_state.twitch_target_aws_account.service_c}"]
  max_size = "${var.max_size}"
  min_size = "${var.min_size}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.app_conf.name}"
  enabled_metrics = [
    "GroupDesiredCapacity",
    "GroupInServiceInstances",
    "GroupMaxSize",
    "GroupMinSize",
    "GroupPendingInstances",
    "GroupStandbyInstances",
    "GroupTerminatingInstances",
    "GroupTotalInstances"
  ]

  tag {
    key = "Name"
    value = "${var.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.project}"
    propagate_at_launch = true
  }
}

resource "aws_autoscaling_policy" "app_asg_scaleout_policy" {
  name = "${var.name}_asg_scaleout_policy"
  scaling_adjustment = 3
  adjustment_type = "ChangeInCapacity"
  cooldown = 1200
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_high_cpu" {
  alarm_name = "${var.name}_asg_high_cpu"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods = "5"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "60"
  statistic = "Average"
  threshold = "50"
  dimensions = {AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"}
  alarm_actions = ["${aws_autoscaling_policy.app_asg_scaleout_policy.arn}"]
}

resource "aws_autoscaling_policy" "app_asg_scalein_policy" {
  name = "${var.name}_asg_scalein_policy"
  scaling_adjustment = -1
  adjustment_type = "ChangeInCapacity"
  cooldown = 900
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_low_cpu" {
  alarm_name = "${var.name}_asg_low_cpu"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods = "15"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "60"
  statistic = "Average"
  threshold = "10"
  dimensions = {AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"}
  alarm_actions = ["${aws_autoscaling_policy.app_asg_scalein_policy.arn}"]
}

module "asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.name}_${var.environment}_topic"
  asg_name = "${aws_autoscaling_group.app_asg.name}"
  lambda_arn = "${data.terraform_remote_state.production_asg_lambda.asg_lambda_arn}"
}

resource "aws_elb" "internal_app_elb" {
  name = "${var.name}-${var.environment}"
  security_groups = ["${var.security_group}"]
  subnets = ["${split(",", "${data.terraform_remote_state.twitch_target_aws_account.service_subnets}")}"]
  internal = true
  cross_zone_load_balancing = true
  connection_draining = true

  listener {
    instance_port = 8000
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  listener {
    instance_port = 8000
    instance_protocol = "http"
    lb_port = 443
    lb_protocol = "https"
    ssl_certificate_id = "${var.https_certificate}"
  }

  health_check {
    # These values are set intentionally low because they are used for graceful deployments
    # Our service waits (unhealthy_threshold + 3) * interval after marking itself as unhealthy
    # and before it restarts during a deployment.
    healthy_threshold = 2
    unhealthy_threshold = 3
    timeout = 2
    target = "HTTP:8000/debug/running"
    interval = 5
  }

  tags {
    Name = "${var.name}-internal-app-elb"
    Owner = "${var.owner}"
    Project = "${var.project}"
  }
}
