resource "aws_sqs_queue" "email_verified_queue" {
  name = "email-verified"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.dead_letter_queue.arn}\",\"maxReceiveCount\":3}"
}

resource "aws_sns_topic_subscription" "email_verification_sqs_subscription" {
  topic_arn = "${lookup(var.email_validation_success_sns, var.environment)}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.email_verified_queue.arn}" 
}

# This policy defines the AWS accounts and their accesss roles for this queue
resource "aws_sqs_queue_policy" "email_verified_queue_policy" {
  queue_url = "${aws_sqs_queue.email_verified_queue.id}"
  policy = <<EOF
  {
    "Version": "2012-10-17",
    "Id": "email_verified_sends_policy",
    "Statement":[
      {
        "Sid": "AllowTopicsFromOtherTwitchAWSAccountsToSendMessages",
        "Effect": "Allow",
        "Principal": "*",
        "Action": "SQS:SendMessage",
        "Resource": "${aws_sqs_queue.email_verified_queue.arn}",
        "Condition": {
          "ArnEquals": {
            "aws:SourceArn": "${lookup(var.email_validation_success_sns, var.environment)}"
          }
        }
      },
      {
        "Sid": "AllowUsersServiceAccountsToAccessMessages",
        "Effect": "Allow",
        "Principal": {
          "AWS": ${jsonencode(split(",", lookup(var.users_service_iam_arns, var.environment)))}
        },
        "Resource": "${aws_sqs_queue.email_verified_queue.arn}",
        "Action": [
          "SQS:ReceiveMessage",
          "SQS:GetQueueUrl",
          "SQS:GetQueueAttributes",
          "SQS:ChangeMessageVisibility",
          "SQS:DeleteMessage"
        ]
      }
    ]
  }
  EOF
}

resource "aws_sqs_queue" "image_upload_queue" {
  name = "image-upload-${var.environment}"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.dead_letter_queue.arn}\",\"maxReceiveCount\":3}"
}

resource "aws_sns_topic_subscription" "image_upload_sqs_subscription" {
  topic_arn = "${aws_sns_topic.user_image_upload_events.arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.image_upload_queue.arn}"
}

resource "aws_sns_topic_subscription" "cache_expiration_sqs_subscription" {
  topic_arn = "${aws_sns_topic.cache_expiration_events.arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.cache_expiration_queue.arn}"
}

resource "aws_sqs_queue" "cache_expiration_queue" {
  name = "expire-cache"
}

resource "aws_sqs_queue_policy" "cache_expiration_queue_policy" {
  queue_url = "${aws_sqs_queue.cache_expiration_queue.id}"
  policy = <<EOF
  {
    "Version": "2012-10-17",
    "Id": "cache_expiration_sends_policy",
    "Statement":[
      {
        "Effect": "Allow",
        "Principal": {
          "AWS": "*"
        },
        "Action": "SQS:SendMessage",
        "Resource": "${aws_sqs_queue.cache_expiration_queue.arn}",
        "Condition": {
          "ArnEquals": {
            "aws:SourceArn": "${aws_sns_topic.cache_expiration_events.arn}"
          }
        }
      },
      {
        "Sid": "AllowUsersServiceAccountsToAccessMessages",
        "Effect": "Allow",
        "Principal": {
          "AWS": ${jsonencode(split(",", lookup(var.users_service_iam_arns, var.environment)))}
        },
        "Resource": "${aws_sqs_queue.cache_expiration_queue.arn}",
        "Action": [
          "SQS:ReceiveMessage",
          "SQS:GetQueueUrl",
          "SQS:GetQueueAttributes",
          "SQS:ChangeMessageVisibility",
          "SQS:DeleteMessage",
          "SQS:SendMessage"
        ]
      }
    ]
  }
  EOF
}
resource "aws_sqs_queue_policy" "image_upload_queue_policy" {
  queue_url = "${aws_sqs_queue.image_upload_queue.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Id": "${aws_sqs_queue.image_upload_queue.arn}/SQSDefaultPolicy",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "*"
      },
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.image_upload_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${aws_sns_topic.user_image_upload_events.arn}"
        }
      }
    },
    {
        "Sid": "AllowUsersServiceAccountsToAccessMessages",
        "Effect": "Allow",
        "Principal": {
          "AWS": ${jsonencode(split(",", lookup(var.users_service_iam_arns, var.environment)))}
        },
        "Resource": "${aws_sqs_queue.image_upload_queue.arn}",
        "Action": [
          "SQS:ReceiveMessage",
          "SQS:GetQueueUrl",
          "SQS:GetQueueAttributes",
          "SQS:ChangeMessageVisibility",
          "SQS:DeleteMessage"
        ]
    }
  ]
}
EOF
}

resource "aws_sqs_queue" "dead_letter_queue" {
  name = "users-service-dead-letter-queue-${var.environment}"
}

resource "aws_sqs_queue_policy" "dead_letter_queue_policy" {
  queue_url = "${aws_sqs_queue.dead_letter_queue.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Id": "${aws_sqs_queue.dead_letter_queue.arn}/SQSDefaultPolicy",
  "Statement": [
    {
        "Sid": "AllowUsersServiceAccountsToAccessMessages",
        "Effect": "Allow",
        "Principal": {
          "AWS": ${jsonencode(split(",", lookup(var.users_service_iam_arns, var.environment)))}
        },
        "Resource": "${aws_sqs_queue.dead_letter_queue.arn}",
        "Action": [
          "SQS:ReceiveMessage",
          "SQS:GetQueueUrl",
          "SQS:GetQueueAttributes",
          "SQS:ChangeMessageVisibility",
          "SQS:DeleteMessage"
        ]
    }
  ]
}
EOF
}
