terraform {
  backend "s3" {
    bucket  = "twitch-users-service-prod"
    key     = "tfstate/web/users-service/terraform/production/us-east-1/beanstalk"
    region  = "us-west-2"
    profile = "twitch-users-service-prod"
  }
}

variable "environment" {
  default = "production"
}

variable "profile" {
  default = "twitch-users-service-prod"
}

variable "region" {
  default = "us-east-1"
}

data "terraform_remote_state" "users-service" {
  backend = "s3"

  config {
    bucket  = "${var.profile}"
    profile = "${var.profile}"
    region  = "us-west-2"
    key     = "tfstate/web/users-service/terraform/${var.environment}/${var.region}"
  }
}

data "terraform_remote_state" "aws_subnets_only" {
  backend = "s3"

  config {
    bucket  = "${var.profile}"
    profile = "${var.profile}"
    region  = "us-west-2"
    key     = "tfstate/web/users-service/terraform/${var.environment}/${var.region}/aws_subnets_only"
  }
}

provider "aws" {
  region  = "${var.region}"
  profile = "${var.profile}"
}

data "aws_iam_policy_document" "users_service_cross_account" {
  statement {
    effect = "Allow"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::641044725657:role/users-service",
      "arn:aws:iam::641044725657:role/user-mutations-stream-role",
    ]
  }
}

resource "aws_iam_role_policy" "users_service_cross_account" {
  name   = "users_service_cross_account"
  policy = "${data.aws_iam_policy_document.users_service_cross_account.json}"

  role = "${data.terraform_remote_state.users-service.iam_role_id}"
}

module "beanstalk" {
  source                  = "../../../modules/beanstalk_single"
  environment             = "${var.environment}"
  aws_region              = "${var.region}"
  new_account_environment = "prod"
  iam_role_id             = "${data.terraform_remote_state.users-service.iam_role_id}"
  iam_role_arn            = "${data.terraform_remote_state.users-service.iam_role_arn}"
  instance_type           = "c4.8xlarge"
  server_asg_max_size     = 60
  server_asg_min_size     = 10
  worker_asg_max_size     = 10
  worker_asg_min_size     = 4
  override_security_group = "${data.terraform_remote_state.aws_subnets_only.sg}"
  platform_version        = "64bit Amazon Linux 2017.09 v2.8.3 running Docker 17.06.2-ce"
}

# twitch-users-service-prod systems state
data "terraform_remote_state" "alarms" {
  backend = "s3"

  config {
    bucket  = "${var.profile}"
    profile = "${var.profile}"
    region  = "us-west-2"
    key     = "tfstate/web/users-service/terraform/${var.environment}/${var.region}/alarms"
  }
}

module "beanstalk_alarms" {
  source                             = "git::git+ssh://git@git-aws.internal.justin.tv/identity/tf_beanstalk.git//alarms?ref=v1.3.0"
  aws_region                         = "${var.region}"
  aws_profile                        = "${var.profile}"
  actions                            = ["${data.terraform_remote_state.alarms.arn}"]
  environment_name                   = "${module.beanstalk.name}"
  application_requests_5xx_threshold = "1000"
}
