// +build integration

package integration

import (
	"context"
	"fmt"
	"os"
	"testing"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/client/usersclient_external"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/tools/integration/tests"
)

type Tester interface {
	Test(t *testing.T)
}

// Relies on the presence of a database containing
// build/*_dump.sql running on localhost:5432,
// dbname="user" username="user" password="user"

func TestIntegration(t *testing.T) {
	serverAddr := os.Getenv("SERVER_ADDRESS")

	if serverAddr == "" {
		t.Fatalf("invalid serverAddr: %q", serverAddr)
	}

	err := image.SetUpImageReader()
	if err != nil {
		logx.Fatal(context.Background(), fmt.Sprint("failed to set up image reader", err))
	}

	c, err := usersclient_internal.NewClient(twitchclient.ClientConf{
		Host: serverAddr,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
	})
	if err != nil {
		t.Fatalf("usersclient_internal.NewClient(%q) err=%q", serverAddr, err)
	}

	cx, err := usersclient_external.NewClient(twitchclient.ClientConf{
		Host: serverAddr,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
	})
	if err != nil {
		t.Fatalf("usersclient_external.NewClient(%q) err=%q", serverAddr, err)
	}

	cc, err := channels.NewClient(twitchclient.ClientConf{
		Host: serverAddr,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
	})
	if err != nil {
		t.Fatalf("channels.NewClient(%q) err=%q", serverAddr, err)
	}

	for _, each := range []struct {
		Name   string
		Tester Tester
	}{
		{
			Name: "TestGetFlow",
			Tester: &tests.TestGetFlow{
				ClientInternal: c,
				ClientExternal: cx,
			},
		},
		{
			Name: "TestBanUserFlow",
			Tester: &tests.TestBanUserFlow{
				Client: c,
			},
		},
		{
			Name: "TestDMCAFlow",
			Tester: &tests.TestDMCAFlow{
				Client: c,
			},
		},
		{
			Name: "TestSetUserFlow",
			Tester: &tests.TestSetUserFlow{
				Client: c,
			},
		},
		{
			Name: "TestRenameFlow",
			Tester: &tests.TestRenameFlow{
				Client: c,
				ChannelClient: cc,
			},
		},
		{
			Name: "TestGetChannelsFlow",
			Tester: &tests.TestGetChannelsFlow{
				Client: cc,
			},
		},
		{
			Name: "TestSetChannelFlow",
			Tester: &tests.TestSetChannelFlow{
				Client: cc,
			},
		},
		{
			Name: "TestCreateUserFlow",
			Tester: &tests.TestCreateUserFlow{
				Client: c,
			},
		},
		{
			Name: "TestDeleteFlow",
			Tester: &tests.TestDeleteFlow{
				Client: c,
			},
		},
	} {
		t.Run(each.Name, each.Tester.Test)
	}
}
