package tests

import (
	"fmt"
	"testing"
	"time"

	"code.justin.tv/web/users-service/client/usersclient_internal"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

type TestBanUserFlow struct {
	Client usersclient_internal.InternalClient
}

func (f *TestBanUserFlow) Test(t *testing.T) {
	testBanUserFlow(t, f.Client)
}

func testGetBannedUsers(t *testing.T, c usersclient_internal.InternalClient, shouldSuccess bool) *models.PropertiesResult {
	untilTime := time.Now().Add(30 * time.Hour)
	props, err := c.GetBannedUsers(context.Background(), untilTime, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}
	if shouldSuccess {
		if props.Results == nil || len(props.Results) != 1 || !Equal((*props.Results[0]).ID, ToBanUserID) {
			assert.FailNow(t, fmt.Sprintf("Didn't get expected banned user in testGetBannedUsers: %s", ToBanUserID))
		}
	} else {
		if len(props.Results) != 0 {
			assert.FailNow(t, fmt.Sprintf("Should get empty props set for testGetBannedUsers, but not: %s", ToBanUserID))
		}
	}
	return props
}

func testBanUserFlow(t *testing.T, c usersclient_internal.InternalClient) {
	testGetBannedUsers(t, c, false)
	prop := &models.BanUserProperties{
		FromUserID:   AdminUserID,
		TargetUserID: ToBanUserID,
		Type:         "tos",
		Reason:       "integration_test",
		Permanent:    false,
		SkipIPBan:    true,
	}
	err := c.BanUserByID(context.Background(), prop, nil)
	require.NoError(t, err)

	props := testGetBannedUsers(t, c, true)
	expectedTosCount := utils.MinValue(utils.IntValue(props.Results[0].TosViolationCount), 2)
	expectedDMCACount := utils.MinValue(utils.IntValue(props.Results[0].DmcaViolationCount), 2)
	err = c.UnbanUserByID(context.Background(), ToBanUserID, true, nil)
	require.NoError(t, err)

	testGetBannedUsers(t, c, false)

	updatedProps, err := c.GetUserByID(context.Background(), ToBanUserID, nil)
	require.NoError(t, err)

	actualTosCount := utils.MinValue(utils.IntValue(updatedProps.TosViolationCount), 2)
	actualDMCACount := utils.MinValue(utils.IntValue(updatedProps.DmcaViolationCount), 2)

	assert.Equal(t, expectedDMCACount, actualDMCACount)
	assert.Equal(t, expectedTosCount, actualTosCount)
}
