package tests

import (
	"fmt"
	"testing"

	"strings"

	"code.justin.tv/common/config"
	"code.justin.tv/web/blacklist"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	bl "code.justin.tv/web/users-service/internal/clients/blacklist"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"
)

func init() {
	config.Register(map[string]string{
		"test-rails-host": "staging-ffa-36-01d094c6-app.dev.us-west2.justin.tv:7000",
	})
}

type TestCreateUserFlow struct {
	Client usersclient_internal.InternalClient
}

func (f *TestCreateUserFlow) Test(t *testing.T) {
	t.Run("TestCreateUser", f.TestCreateUser)
	t.Run("TestCreateUserWithBlacklistedIP", f.TestCreateUserWithBlacklistedIP)
}

func (f *TestCreateUserFlow) TestCreateUser(t *testing.T) {
	newUserlogin := "newUserlogin"
	defaultCategory := "gaming"
	defaultLanguage := "en"
	up := &models.CreateUserProperties{
		Login:    newUserlogin,
		Birthday: GetBirthday(AdminUserID),
		Email:    *Users[AdminUserID].Email,
	}

	prop, err := f.Client.CreateUser(context.Background(), up, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testCreateUser: %s", err, newUserlogin))
	}
	prop.ProfileImage = nil

	lowerLogin := strings.ToLower(newUserlogin)
	location := ""
	newUser := &models.Properties{
		ID:                  prop.ID,
		Login:               &lowerLogin,
		Birthday:            Users[AdminUserID].Birthday,
		Email:               Users[AdminUserID].Email,
		Language:            &defaultLanguage,
		Category:            &defaultCategory,
		UpdatedOn:           prop.UpdatedOn,
		CreatedOn:           prop.CreatedOn,
		Displayname:         &newUserlogin,
		RemoteIP:            prop.RemoteIP,
		Location:            &location,
		LastLoginChangeDate: prop.LastLoginChangeDate,
	}

	if !Equal(prop, newUser) {
		assert.FailNow(t, fmt.Sprintf("Property not equal for testCreateUser: %s", newUserlogin))
	}
}

func (f *TestCreateUserFlow) TestCreateUserWithBlacklistedIP(t *testing.T) {
	newUserlogin := "newUserlogin2"
	ip := "123.123.123.123"
	up := &models.CreateUserProperties{
		Login:    newUserlogin,
		Birthday: GetBirthday(AdminUserID),
		Email:    *Users[AdminUserID].Email,
		IP:       ip,
	}

	bc, err := configureBlacklistClient()
	err = bc.AddIP(context.Background(), ip, "", nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testCreateUserWithBlacklistedIP: %s", err, newUserlogin))
	}

	b, err := bc.ValidateIP(context.Background(), ip, nil)
	if !b || err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testCreateUserWithBlacklistedIP: %s", err, newUserlogin))
	}

	_, err = f.Client.CreateUser(context.Background(), up, nil)
	if err != models.ErrIPBlacklisted {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testCreateUserWithBlacklistedIP: %s", err, newUserlogin))
	}
}

func configureBlacklistClient() (blacklist.Client, error) {
	return &bl.NoopClient{}, nil
}
