package tests

import (
	"fmt"
	"testing"

	"strings"

	"reflect"

	"code.justin.tv/web/users-service/client/usersclient_external"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

type TestGetFlow struct {
	ClientInternal usersclient_internal.InternalClient
	ClientExternal usersclient_external.ExternalClient
}

func (f *TestGetFlow) Test(t *testing.T) {
	t.Run("TestGetUserByIDSuccess", f.TestGetUserByIDSuccess)
	t.Run("TestGetUserByLoginSuccess", f.TestGetUserByLoginSuccess)
	t.Run("TestGetUsersSuccess", f.TestGetUsersSuccess)
	t.Run("TestGetUsersInlineIdentifierValidationSuccess", f.TestGetUsersInlineIdentifierValidationSuccess)
	t.Run("TestGetUserByEmailSuccess", f.TestGetUserByEmailSuccess)
	t.Run("TestGetUsersByLoginLikePrefixSuccess", f.TestGetUsersByLoginLikePrefixSuccess)
	t.Run("TestGetGlobalPrivilegedUsersSuccess", f.TestGetGlobalPrivilegedUsersSuccess)
	t.Run("TestGetTypeForLoginSuccess", f.TestGetTypeForLoginSuccess)
	t.Run("TestGetUserByIDNonAuthedSuccess", f.TestGetUserByIDNonAuthedSuccess)
	t.Run("TestGetUserByLoginNonAuthedSuccess", f.TestGetUserByLoginNonAuthedSuccess)
	t.Run("TestGetUsersNonAuthedSuccess", f.TestGetUsersNonAuthedSuccess)
}

func (f *TestGetFlow) TestGetUserByIDSuccess(t *testing.T) {
	props, err := f.ClientInternal.GetUserByID(context.Background(), testutils.AdminUserID, nil)
	require.NoError(t, err)

	testutils.AssertEqual(t, testutils.Users[testutils.AdminUserID], props)
}

func (f *TestGetFlow) TestGetUserByLoginSuccess(t *testing.T) {
	props, err := f.ClientInternal.GetUserByLogin(context.Background(), *testutils.Users[testutils.AdminUserID].Login, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	testutils.AssertEqual(t, testutils.Users[testutils.AdminUserID], props)
}

func (f *TestGetFlow) TestGetUsersSuccess(t *testing.T) {
	params := &models.FilterParams{
		IDs: []string{testutils.AdminUserID, testutils.ToBanUserID, testutils.ToRenameUserID, testutils.ToSetPropUserID, testutils.ToAddDMCAUserID},
	}
	props, err := f.ClientInternal.GetUsers(context.Background(), params, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	for i := range props.Results {
		user := testutils.Users[(*props.Results[i]).ID]
		testutils.AssertEqual(t, user, props.Results[i])
	}
}

func (f *TestGetFlow) TestGetUsersInlineIdentifierValidationSuccess(t *testing.T) {
	params := &models.FilterParams{
		IDs: []string{testutils.AdminUserID, testutils.ToBanUserID, testutils.ToRenameUserID, testutils.ToSetPropUserID, testutils.ToAddDMCAUserID, "test"},
		InlineIdentifierValidation: true,
	}
	props, err := f.ClientInternal.GetUsers(context.Background(), params, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	if len(props.BadIdentifiers) != 1 {
		assert.FailNow(t, fmt.Sprintf("Expected 1 error, got %d", len(props.BadIdentifiers)))
	}

	for i := range props.Results {
		user := testutils.Users[(*props.Results[i]).ID]
		testutils.AssertEqual(t, user, props.Results[i])
	}
}

func (f *TestGetFlow) TestGetUserByEmailSuccess(t *testing.T) {
	props, err := f.ClientInternal.GetUserByEmail(context.Background(), *testutils.Users[testutils.AdminUserID].Email, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	testutils.AssertEqual(t, testutils.Users[testutils.AdminUserID], props)
}

func (f *TestGetFlow) TestGetUsersByLoginLikePrefixSuccess(t *testing.T) {
	props, err := f.ClientInternal.GetUsersByLoginLike(context.Background(), testutils.LoginLikePrefixPattern, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	for i := range props.Results {
		login := (*props.Results[i]).Login
		loginLikePrefix := testutils.LoginLikePrefixPattern[:len(testutils.LoginLikePrefixPattern)-1]
		if !strings.HasPrefix(*login, loginLikePrefix) {
			assert.FailNow(t, fmt.Sprintf("Login Prefix not equal for testGetUsersByLoginLikePrefixSuccess: %s", *login))
		}
	}
}

func (f *TestGetFlow) TestGetGlobalPrivilegedUsersSuccess(t *testing.T) {
	logins, err := f.ClientInternal.GetGlobalPrivilegedUsers(context.Background(), nil, nil)
	require.NoError(t, err)
	require.NotNil(t, logins)

	if !reflect.DeepEqual(*logins, *testutils.PrivilegedUsers) {
		assert.FailNow(t, fmt.Sprintf("Logins not equal for testGetGlobalPrivilegedUsersSuccess: %s", *logins))
	}
}

func (f *TestGetFlow) TestGetTypeForLoginSuccess(t *testing.T) {
	props, err := f.ClientInternal.GetTypeForLogin(context.Background(), *testutils.Users[testutils.AdminUserID].Login, "", false, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	testutils.AssertEqual(t, props.Login, *testutils.Users[testutils.AdminUserID].Login)
}

func (f *TestGetFlow) TestGetUserByIDNonAuthedSuccess(t *testing.T) {
	props, err := f.ClientExternal.GetUserByID(context.Background(), testutils.AdminUserID, testutils.ToBanUserID, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	testutils.AssertEqual(t, testutils.Users[testutils.AdminUserID].ConvertToExternal(), props)
}

func (f *TestGetFlow) TestGetUserByLoginNonAuthedSuccess(t *testing.T) {
	props, err := f.ClientExternal.GetUserByLogin(context.Background(), *testutils.Users[testutils.AdminUserID].Login, testutils.ToBanUserID, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	testutils.AssertEqual(t, testutils.Users[testutils.AdminUserID].ConvertToExternal(), props)
}

func (f *TestGetFlow) TestGetUsersNonAuthedSuccess(t *testing.T) {
	params := &models.FilterParams{
		IDs: []string{testutils.AdminUserID, testutils.ToBanUserID, testutils.ToRenameUserID, testutils.ToSetPropUserID, testutils.ToAddDMCAUserID},
	}
	props, err := f.ClientExternal.GetUsers(context.Background(), testutils.ToBanUserID, params, nil)
	require.NoError(t, err)
	if props == nil {
		t.Fatal("props is nil")
	}

	for i := range props.Results {
		user := testutils.Users[(*props.Results[i]).ID]
		testutils.AssertEqual(t, user.ConvertToExternal(), props.Results[i])
	}
}
