package tests

import (
	"fmt"
	"testing"
	"time"

	"strings"

	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

var newLogin = "yourNewName"

type TestRenameFlow struct {
	Client        usersclient_internal.InternalClient
	ChannelClient channels.Client
}

func (f *TestRenameFlow) Test(t *testing.T) {
	testRenameFlow(t, f.Client, f.ChannelClient)
}

func testGetRenameEligibility(t *testing.T, c usersclient_internal.InternalClient, ifAbleToRename bool) {
	props, err := c.GetRenameEligibility(context.Background(), Users[ToRenameUserID].ID, nil)
	require.NoError(t, err)

	if ifAbleToRename {
		if !*props.RenameEligible && (*props.RenameEligibleOn).After(time.Now()) {
			assert.FailNow(t, fmt.Sprintf("Properties not correct for testGetRenameEligibility: %s", ToRenameUserID))
		}
	} else {
		if *props.RenameEligible && (*props.RenameEligibleOn).Before(time.Now()) {
			assert.FailNow(t, fmt.Sprintf("Properties not correct for testGetRenameEligibility: %s", ToRenameUserID))
		}
	}
}

func testRename(c usersclient_internal.InternalClient, name string, skipCooldown bool) error {
	uup := &models.UpdateableProperties{
		NewLogin:          &name,
		SkipLoginCooldown: &skipCooldown,
	}

	return c.SetUser(context.Background(), ToRenameUserID, uup, nil)
}

func testRenameFlow(t *testing.T, c usersclient_internal.InternalClient, cc channels.Client) {
	// Rename user
	testGetRenameEligibility(t, c, true)
	err := testRename(c, newLogin, false)
	require.NoError(t, err)

	err = safeExpire(context.Background(), c, ToSetPropUserID)
	if err != nil {
		assert.Fail(t, fmt.Sprintf("Error %v cannot expire cache for: %s during rename testing", err, ToSetPropUserID))
	}

	// Check the redirect channel field has been changed
	channelWithRedirect, err := cc.GetByLogin(context.Background(), Channels[ToSetPropUserID].Name, nil)
	assert.Equal(t, *(*channelWithRedirect).RedirectChannel, strings.ToLower(newLogin))

	// Rename again
	testGetRenameEligibility(t, c, false)
	err = testRename(c, ToRenameUserID, false)
	if err == nil {
		assert.FailNow(t, fmt.Sprintf("Properties not correct for testRenameFlow: %s", ToRenameUserID))
	}

	// Rename back skipping cooldown
	testGetRenameEligibility(t, c, false)
	err = testRename(c, ToRenameUserID, true)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v encountered for testRenameFlow with skipped cooldown: %s", err, ToRenameUserID))
	}
}
