package tests

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/web/users-service/client/channels"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
)

type TestSetChannelFlow struct {
	Client channels.Client
}

func (f *TestSetChannelFlow) Test(t *testing.T) {
	t.Run("TestSetChannel", f.TestSetChannel)
	t.Run("TestSetChannelWithOnlyBasicInfo", f.TestSetChannelWithOnlyBasicInfo)
}

func (f *TestSetChannelFlow) TestSetChannel(t *testing.T) {
	up := models.UpdateChannelProperties{
		ID:                  Channels[AdminUserID].ID,
		BlockNonPublicAds:   GetBoolPointer(true),
		FightAdBlock:        GetBoolPointer(true),
		DirectoryHidden:     GetBoolPointer(true),
		BroadcasterLanguage: GetStringPointer("ko"),
		BroadcasterSoftware: GetStringPointer("bala"),
		Mature:              GetBoolPointer(true),
		About:               GetStringPointer("test"),
		DisableChat:         GetBoolPointer(true),
	}

	channel, err := f.Client.Get(context.Background(), AdminUserID, nil)
	if err != nil {
		t.Fatal("failed to set channel with correct property:", err)
	}

	if !ChannelEqual(Channels[AdminUserID], *channel) {
		assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetChannel: %s", AdminUserID))
	}

	err = f.Client.Set(context.Background(), up, nil)
	if err != nil {
		t.Fatal("failed to set channel:", err)
	}

	channel, err = f.Client.Get(context.Background(), AdminUserID, nil)
	if err != nil {
		t.Fatal("failed to set channel with correct property:", err)
	}

	newChannel := CopyChannel(Channels[AdminUserID])
	newChannel.BlockNonPublicAds = GetBoolPointer(true)
	newChannel.FightAdBlock = GetBoolPointer(true)
	newChannel.DirectoryHidden = true
	newChannel.BroadcasterLanguage = GetStringPointer("ko")
	newChannel.BroadcasterSoftware = GetStringPointer("bala")
	newChannel.Mature = GetBoolPointer(true)
	newChannel.About = GetStringPointer("test")
	newChannel.DisableChat = GetBoolPointer(true)
	newChannel.UpdatedOn = &channel.UpdatedOn

	if !ChannelEqual(newChannel, *channel) {
		assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetChannel: %s", AdminUserID))
	}
}

func (f *TestSetChannelFlow) TestSetChannelWithOnlyBasicInfo(t *testing.T) {
	up := models.UpdateChannelProperties{
		ID:                  Channels[ToSetChannelPropUserID].ID,
		BlockNonPublicAds:   GetBoolPointer(true),
		FightAdBlock:        GetBoolPointer(true),
		DirectoryHidden:     GetBoolPointer(true),
		BroadcasterLanguage: GetStringPointer("ko"),
		BroadcasterSoftware: GetStringPointer("bala"),
		Mature:              GetBoolPointer(true),
		About:               GetStringPointer("test"),
		DisableChat:         GetBoolPointer(true),
	}

	oldchannel, err := f.Client.Get(context.Background(), ToSetChannelPropUserID, nil)
	if err != nil {
		t.Fatal("failed to set channel with correct property:", err)
	}

	err = f.Client.Set(context.Background(), up, nil)
	if err != nil {
		t.Fatal("failed to set channel:", err)
	}

	channel, err := f.Client.Get(context.Background(), ToSetChannelPropUserID, nil)
	if err != nil {
		t.Fatal("failed to set channel with correct property:", err)
	}

	oldchannel.BlockNonPublicAds = true
	oldchannel.FightAdBlock = true
	oldchannel.DirectoryHidden = true
	oldchannel.BroadcasterLanguage = "ko"
	oldchannel.BroadcasterSoftware = "bala"
	oldchannel.Mature = true
	oldchannel.About = "test"
	oldchannel.DisableChat = true
	oldchannel.UpdatedOn = channel.UpdatedOn

	if !Equal(oldchannel, channel) {
		assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetChannel: %s", ToSetChannelPropUserID))
	}
}
