package tests

import (
	"fmt"
	"testing"
	"time"

	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/internal/image"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"
)

var newEmail = "new_email@hotmail.com"

type TestSetUserFlow struct {
	Client usersclient_internal.InternalClient
}

func (f *TestSetUserFlow) Test(t *testing.T) {
	t.Run("TestSetUserEmailAndImage", f.TestSetUserEmailAndImage)
	t.Run("TestSetUserPhoneNumber", f.TestSetUserPhoneNumber)
}

func (f *TestSetUserFlow) TestSetUserEmailAndImage(t *testing.T) {
	lastLogin := time.Now().Format(validators.LastLoginFormat)
	uup := &models.UpdateableProperties{
		ID:        ToSetPropUserID,
		Email:     &newEmail,
		LastLogin: &lastLogin,
	}
	err := f.Client.SetUser(context.Background(), ToSetPropUserID, uup, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testSetUserEmail: %s", err, ToSetPropUserID))
	}

	props, err := f.Client.GetUserByID(context.Background(), ToSetPropUserID, nil)
	updatedUser := CopyUser(Users[ToSetPropUserID])
	updatedUser.Email = &newEmail
	updatedUser.UpdatedOn = props.UpdatedOn
	updatedUser.LastLogin = &lastLogin

	//if !Equal(props, updatedUser) {
	//	assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetUserEmail: %s", ToSetPropUserID))
	//}

	// Test Set image
	imageUpdates := models.ImageProperties{
		ID:           ToSetPropUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	err = f.Client.SetUserImageMetadata(context.Background(), imageUpdates, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testSetUserImageMetadata: %s", err, ToSetPropUserID))
	}

	props, err = f.Client.GetUserByID(context.Background(), ToSetPropUserID, nil)

	imageString, err := yimg.ProfileImagesToString(Users[AdminUserID].ProfileImage)
	images, err := yimg.ProfileImages([]byte(*imageString), *Users[ToSetPropUserID].Login)
	updatedUser.ProfileImage = &images
	updatedUser.UpdatedOn = props.UpdatedOn
	updatedUser.LastLogin = &lastLogin
	emailVerified := false
	updatedUser.EmailVerified = &emailVerified

	if !Equal(props, updatedUser) {
		assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetUserEmailAndImage: %s", ToSetPropUserID))
	}

	// Test Set Default image
	imageUpdates = models.ImageProperties{
		ID: ToSetPropUserID,
	}

	for key, val := range image.DefaultProfileImages {
		imageUpdates.DefaultProfileImage = &key
		updatedUser.ProfileImage = val
		break
	}

	err = f.Client.SetUserImageMetadata(context.Background(), imageUpdates, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testSetUserImageMetadata: %s", err, ToSetPropUserID))
	}

	props, err = f.Client.GetUserByID(context.Background(), ToSetPropUserID, nil)
	updatedUser.UpdatedOn = props.UpdatedOn

	if !Equal(props, updatedUser) {
		assert.FailNow(t, fmt.Sprintf("Properties not equal for testSetUserEmailAndImage: %s", ToSetPropUserID))
	}
}

func (f *TestSetUserFlow) TestSetUserPhoneNumber(t *testing.T) {

	user := PhoneNumberUserID
	code := "123123"
	err := f.Client.VerifyUserPhoneNumber(context.Background(), user, code, nil)
	if err != models.ErrVerifyCodeExpired {
		t.Fatalf("expected %s, got %s", models.ErrVerifyCodeExpired, err)
	}

	number := "1231231234"

	uup := &models.UpdateableProperties{
		PhoneNumber: &number,
	}
	err = f.Client.SetUser(context.Background(), user, uup, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testSetUserPhoneNumber: %s", err, user))
	}

	props, err := f.Client.GetUserByID(context.Background(), user, nil)
	assert.Nil(t, err)

	if props.PhoneNumber == nil {
		t.Fatal("expected phone number to not be nil")
	}

	err = f.Client.VerifyUserPhoneNumber(context.Background(), user, code, nil)
	if err != nil {
		assert.FailNow(t, fmt.Sprintf("Error %v in for testSetUserPhoneNumber: %s", err, user))
	}

	props, err = f.Client.GetUserByID(context.Background(), user, nil)
	assert.Nil(t, err)

	if props.PhoneNumber == nil {
		t.Fatal("expected phone number to not be nil")
		return
	}

	assert.Equal(t, number, *props.PhoneNumber)
}
